#!/bin/bash

# This script will get the repo XML file from googles server
# This same file is used by Google Android's SDK Manager
# It:
#  - extracts all necessary information to file "version_list.txt"
#  - extracts the licenses. They are store in *.copyright files in the dir of this scripts
#  - creates package.xml files
#  - builds debian/control file
#  - builds debian/tests/control file
#  - inform on the timestamp in the XML file

SCRIPT_PATH=$(dirname $0)

cd $SCRIPT_PATH

TMPDIR=$(mktemp -d)
# Download upstream XML
#SERVER_URL=https://raw.githubusercontent.com/eagletmt/android-repository-history/master/repository
SERVER_URL=https://dl.google.com/android/repository/
REPO_DIR=
UPSTREAM_XML=repository2-1.xml
wget ${SERVER_URL}${REPO_DIR}${UPSTREAM_XML} -O $TMPDIR/$(basename $UPSTREAM_XML)

GENERATION_DATE=$(grep "Generated on" $TMPDIR/$(basename $UPSTREAM_XML) | sed "s/.*Generated on \(.*\) with.*/\1/")
TIMESTAMP=$(date --date="$GENERATION_DATE" +%s)
echo "TIMESTAMP=$TIMESTAMP" > ../version_list.info
echo "TIMESTAMP_DATE=\"$(date -Iseconds -u -d@$TIMESTAMP)\"" >> ../version_list.info
echo "REPO_DIR=$REPO_DIR" >> ../version_list.info

# Extract info from XML to version_list.txt
xsltproc list_available_packages.xsl "$TMPDIR/$(basename $UPSTREAM_XML)" >> "$TMPDIR/version_list.txt"

# Manually add some lines
echo "ndk;10.4		10.4	NDK (Side by side) r10e	android-sdk-license	1110915721	android-ndk-r10e-linux-x86_64.zip	f692681b007071103277f6edc6f91cb5c5494a32		patcher;v4	" >> "$TMPDIR/version_list.txt"
if grep --perl-regexp "extras;google;auto\t\t2\.0.*77e3f80c2834e1fad33f56539ceb0215da408fab" "$TMPDIR/version_list.txt" > /dev/null ; then
    # Fix version of Android Auto to 2.0.0+really2.0 because upstream labeled 2.0rc2 as 2.0.2, and the final
    # release is labeled 2.0
    sed -i "s/extras;google;auto\t\t2\.0\t/extras;google;auto\t\t2.0.2+really2.0\t/" "$TMPDIR/version_list.txt"
fi
# Starting from 2023-01, package for platforms;android-33 rev1 (filename platform-33-ext4_r01.zip) appeared
# while rev2 is already there. Google's sdkmanager only proposes rev2, so remove rev1 from the files.
sed -i "/platforms;android-33\t33\t1\tAndroid SDK Platform 33\tandroid-sdk-license\t67564453\tplatform-33-ext4_r01.zip\t422f50033523c5f934f38df520699080b59bbc7e/ d"  "$TMPDIR/version_list.txt"

# Remove package.xml files
for dir in $(grep -v "^#" "$TMPDIR/version_list.txt" | cut -d "	" -f1 | cut -d ";" -f1 | sort -u); do
    rm -fr ../../$dir
done

# Create package.xml files
xsltproc make_package.xml.xsl "$TMPDIR/$(basename $UPSTREAM_XML)"

# Add unpacked size of components to version_list_all.txt
LC_ALL=C join --check-order -a1 -t $'\t'  <(LC_ALL=C sort "$TMPDIR/version_list.txt") <(LC_ALL=C sort "../version_list_unpacked_size.txt") > "$TMPDIR/version_list_all.txt"

# Write everything to ../version_list.txt
echo "# This file is generated automatically by script in debian/scripts/$(basename $0)" > ../version_list.txt
echo "# Don't update this file manually." >> "../version_list.txt"
sort -V "$TMPDIR/version_list_all.txt" >> "../version_list.txt"

# Extract licenses
rm -r ../licenses
mkdir -p ../licenses
rm -r ../../licenses
mkdir -p ../../licenses
for license in $(grep -v "^#" ../version_list.txt | cut -d "	" -f 5 | sort -u); do
    xsltproc --stringparam license $license extract-copyrights.xsl "$TMPDIR/$(basename $UPSTREAM_XML)" > ../licenses/$license.copyright
    # Compute sha1 of license (we need to remove newline at end of file to be consistent with:
    # https://android.googlesource.com/platform/tools/base/+/gradle_2.2.2/repository/src/main/java/com/android/repository/api/License.java)
    perl -p -e 'chomp if eof' ../licenses/$license.copyright | sha1sum | cut -d " " -f 1 > ../../licenses/$license
    # Reformat for inclusion in copyright file
    sed -i -e "s/^$/./" -e "s/^/ /" ../licenses/$license.copyright
done

rm -r $TMPDIR

./build_control_file.sh
./build_tests_control_file.sh

cd ../..
if dpkg --compare-versions $TIMESTAMP gt $(dpkg-parsechangelog --show-field Version); then
    echo "Generation Date of XML: $GENERATION_DATE"
    echo "Timestamp: $TIMESTAMP"
    echo "You may update version in changelog with that version with"
    echo "  dch -v $TIMESTAMP"
else
    echo "XML generation timestamp is not newer than last version in debian/changelog"
fi
