/*
 * Copyright (C) 2000-2024 the xine project
 *
 * This file is part of xine, a unix video player.
 *
 * xine is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * xine is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110, USA
 *
 */

#ifndef __COMMON_H__
#define __COMMON_H__

#ifndef PACKAGE_NAME
#error config.h not included
#endif

#include <stdio.h>

#include <xine.h>
#include <xine/xineutils.h>
#include <xine/sorted_array.h>

typedef struct gGui_st gGui_t;

#include "xitk.h"
#include "dlist.h"

typedef struct xui_keyedit_s xui_keyedit_t;
typedef struct xui_vwin_st xui_vwin_t;
typedef struct xui_actrl_st xui_actrl_t;
typedef struct xui_vctrl_st xui_vctrl_t;
typedef struct xui_event_sender_st xui_event_sender_t;
typedef struct xui_help_s xui_help_t;
typedef struct xui_mrlb_st xui_mrlb_t;
typedef struct xui_panel_st xui_panel_t;
typedef struct xui_mmkedit_s xui_mmkedit_t;
typedef struct xui_playlist_st xui_playlist_t;
typedef struct xui_setup_st xui_setup_t;
typedef struct xui_viewlog_s xui_viewlog_t;
typedef struct filebrowser_s filebrowser_t;
typedef struct xui_tvset_s xui_tvset_t;
typedef struct xui_tvout_s xui_tvout_t;
typedef struct xui_skdloader_s xui_skdloader_t;
typedef struct xui_sinfo_s xui_sinfo_t;
typedef struct xui_lirc_s xui_lirc_t;
typedef struct xui_network_remote_s xui_network_remote_t;

#include "config_wrapper.h"
#include "i18n.h"
#include "post.h"
#include "osd.h"
#include "kbindings.h"
#include "mediamark.h"

#include "libcommon.h"

#ifdef HAVE_ORBIT
#include "../corba/xine-server.h"
#endif

/*
 * config related constants
 */
#define CONFIG_LEVEL_BEG         0 /* => beginner */
#define CONFIG_LEVEL_ADV        10 /* advanced user */
#define CONFIG_LEVEL_EXP        20 /* expert */
#define CONFIG_LEVEL_MAS        30 /* motku */
#define CONFIG_LEVEL_DEB        40 /* debugger (only available in debug mode) */

#define CONFIG_NO_DESC          NULL
#define CONFIG_NO_HELP          NULL
#define CONFIG_NO_CB            NULL
#define CONFIG_NO_DATA          NULL

/* Sound mixer capabilities */
#define MIXER_CAP_NOTHING       0x00000000
#define MIXER_CAP_VOL           0x00000001
#define MIXER_CAP_MUTE          0x00000002

/* mixer control method */
typedef enum {
  SOUND_CARD_MIXER = 0,
  SOFTWARE_MIXER,
  LAST_MIXER
} xui_mixer_type_t;

/* Playlist loop modes */
typedef enum {
  PLAYLIST_LOOP_NO_LOOP = 0, /* no loop (default) */
  PLAYLIST_LOOP_LOOP,        /* loop the whole playlist */
  PLAYLIST_LOOP_REPEAT,      /* loop the current mrl */
  PLAYLIST_LOOP_SHUFFLE,     /* random selection in playlist */
  PLAYLIST_LOOP_SHUF_PLUS,   /* random selection in playlist, never ending */
  PLAYLIST_LOOP_MODES_NUM
} playlist_loop_t;

#define PLAYLIST_CONTROL_STOP         0x00000001 /* Don't start next entry in playlist */
#define PLAYLIST_CONTROL_STOP_PERSIST 0x00000002 /* Don't automatically reset PLAYLIST_CONTROL_STOP */
#define PLAYLIST_CONTROL_IGNORE       0x00000004 /* Ignore some playlist adding action */

#define SAFE_FREE(x)            do {           \
                                    free((x)); \
                                    x = NULL;  \
                                } while(0)

#define ABORT_IF_NULL(p)                                                                      \
  do {                                                                                        \
    if((p) == NULL) {                                                                         \
      fprintf(stderr, "%s(%d): '%s' is NULL. Aborting.\n",  __XINE_FUNCTION__, __LINE__, #p); \
      abort();                                                                                \
    }                                                                                         \
  } while(0)

/* Our default location for skin downloads */
#define SKIN_SERVER_URL         "http://xine.sourceforge.net/skins/skins.slx"

#define fontname       "-*-helvetica-medium-r-*-*-12-*-*-*-*-*-*-*"
#define boldfontname   "-*-helvetica-bold-r-*-*-11-*-*-*-*-*-*-*"
#define hboldfontname  "-*-helvetica-bold-r-*-*-12-*-*-*-*-*-*-*"
#define lfontname      "-*-helvetica-bold-r-*-*-11-*-*-*-*-*-*-*"
#define tabsfontname   "-*-helvetica-bold-r-*-*-12-*-*-*-*-*-*-*"
#define br_fontname    "-misc-fixed-medium-r-normal-*-12-*-*-*-*-*-*-*"
#define btnfontname    "-*-helvetica-bold-r-*-*-12-*-*-*-*-*-*-*"

/* easy pseudo widgets for *_main calls. see the .h files for suport. */
#define XUI_W_TOGGLE NULL
#define XUI_W_ON ((xitk_widget_t *)1)
#define XUI_W_OFF ((xitk_widget_t *)2)
#define XUI_W_INIT ((xitk_widget_t *)3)
#define XUI_W_DEINIT ((xitk_widget_t *)4)

typedef struct skins_locations_s skins_locations_t;

typedef enum {
  XUI_EXTS_ANY = 0,
  XUI_EXTS_MEDIA,
  XUI_EXTS_AUDIO,
  XUI_EXTS_VIDEO,
  XUI_EXTS_IMAGE,
  XUI_EXTS_SUBTITLE,
  XUI_EXTS_PLAYLIST,
  XUI_EXTS_LAST
} xui_exts_t;

struct gGui_st {
  struct gGui_st           *nextprev[3];

  char                     *cfg_file;
  xine_t                   *xine;
  xine_video_port_t        *vo_port;
  xine_video_port_t        *vo_none;
  xine_audio_port_t        *ao_none;

  int                       verbosity;
  int                       post_video_enable;
  int                       post_audio_enable;
  int                       deinterlace_enable;
  int                       lirc_enable;

  xine_audio_port_t        *ao_port;

  xine_stream_t            *stream;

  int                       subtitle_autoload;
  xine_stream_t            *spu_stream;

  int                       broadcast_port;

  struct {
    /* Seek through xine_play () may be slow. We defer it to a separate thread. */
    pthread_mutex_t         mutex;
    int                     running; /** << 0 = no, 1 = yes, 2 = shutdown. */
    int                     newpos;
    int                     timestep;
    int                     pos_time_length[3];
  }                         seek;

  int                       smart_mode;

  xine_event_queue_t       *event_queue;

  /* Visual stuff (like animation in video window while audio only playback) */
  struct {
    xine_stream_t          *stream;
    xine_event_queue_t     *event_queue;
    int                     running;
    int                     current;
    int                     enabled; /* 0, 1:vpost, 2:vanim */

    int                     num_mrls;
    char                  **mrls;

    int                     post_plugin_num;
    int                     post_changed;
    post_element_t          post_output_element;
  } visual_anim;

  post_info_t               post_audio, post_video;

  struct {
    pthread_mutex_t         mutex;

    int                     initialized;
    int                     enabled;
    int                     use_unscaled;
    int                     unscaled_available;
    int                     timeout;

    int                     last_shown_time;

    xitk_dlist_t            pending;
    struct xui_osd_obj_s {
      xitk_dnode_t          node;
      osd_object_t          obj;
    }                       sinfo, bar, status, info;

  } osd;

  int                       experience_level;

  int                       logo_mode;
  int                       logo_has_changed;
  int                       display_logo;
  const char               *logo_mrl;

  /* stuff like ACTID_x */
#define MAX_ACTIONS_ON_START 32
  action_id_t               actions_on_start[MAX_ACTIONS_ON_START+1];
  char                     *autoscan_plugin;

  xitk_t                   *xitk;

  /* basic X11 stuff */
  xitk_image_t             *icon;

  xitk_skin_config_t       *skin_config;
  char                     *skin_server_url;

  int                       cursor_visible;
  int                       cursor_grabbed;

  uint32_t                  debug_level;

  int                       is_display_mrl; /** << show 0 (ident), 1 (mrl) */
  int                       runtime_mode;   /** << show 0 (elapsed time), 1 (remaining time) */

  /* Current mediamark */
  mediamark_t               mmk;
  /* Recursive mutex, protecting .mmk and .playlist. */
  pthread_mutex_t           mmk_mutex;

  /* playlist */
  struct {
    mediamark_t           **mmk;

    int                     max;        /** << number of entries that fit .mmk */
    int                     num;        /** << number of entries set in .mmk */
    int                     cur;        /** << current entry in playlist */
    int                     ref_append; /** << append mrl reference entry to this position */
    playlist_loop_t         loop;       /** << see PLAYLIST_LOOP_* */
    int                     control;    /** << see PLAYLIST_CONTROL_* */
    uint32_t                gen;        /** << up 1 for each change, except .cur */
    int                     played;     /** << number of entries played (0...num) */
    int                     probed;     /** << number of entries played at least 1 time or probed (0...num) */
    uint32_t                probe_mode; /** << | 1 (probe files), | 2 (probe network mrls) */
    uint32_t                probe_gen;  /** << avoid useless reprobe */

    xine_sarray_t          *exts[XUI_EXTS_LAST];
  } playlist;


  int                       on_quit;
  int                       running;
  int                       ignore_next;

  int                       stdctl_enable;

  struct {
    xui_mixer_type_t        type_volume, type_mute;
    int                     mute[LAST_MIXER];
    int                     level[LAST_MIXER]; /** << -1 if not supported */
  } mixer;

  struct {
    const char             *msg[4];
    int                     flags;
  } transform;

  int                       layer_above;

  int                       network;
  int                       network_port;
  xui_network_remote_t     *network_remote;

  int                       use_root_window;

  int                       ssaver_enabled;
  int                       ssaver_timeout;

  int                       skip_by_chapter;

  int                       auto_vo_visibility;
  int                       auto_panel_visibility;

  const char               *snapshot_location;

  char                      *keymap_file;
  kbinding_t                *kbindings;
  int                        shortcut_style;
  int                        kbindings_enabled;

  /* event handling */
  struct {
    int                      set;
    int                      arg;
  } numeric;
  struct {
    int                      set;
    char                    *arg;
  } alphanum;
  int                        event_reject;
  int                        event_pending;
  pthread_mutex_t            event_mutex;
  pthread_cond_t             event_safe;

  int                        eventer_sticky;
  int                        stream_info_auto_update;

  char                       curdir[XITK_PATH_MAX];

  int                        play_anyway;

  int                        splash;

  pthread_mutex_t            download_mutex;

  FILE                      *report;

  int                        no_gui;
  int                        no_mouse;

  unsigned int               gfx_saturation;

  void                     (*nongui_error_msg) (gGui_t *gui, const char *text);

  FILE                      *orig_stdout; /* original stdout at startup        */
                                          /* before an evtl. later redirection */

  xitk_window_t             *splash_win;
  xui_panel_t               *panel;
  xui_vwin_t                *vwin;
  xui_setup_t               *setup;
  xui_mrlb_t                *mrlb;
  xui_actrl_t               *actrl;
  xui_vctrl_t               *vctrl;
  xui_event_sender_t        *eventer;
  xui_skdloader_t           *skdloader;
  xui_help_t                *help;
  xui_viewlog_t             *viewlog;
  xui_keyedit_t             *keyedit;
  xui_playlist_t            *plwin;
  xui_mmkedit_t             *mmkedit;
  xui_sinfo_t               *streaminfo;
  xui_tvset_t               *tvset;
  xui_tvout_t               *tvout;
  xui_lirc_t                *lirc;
  filebrowser_t             *pl_load;
  filebrowser_t             *pl_save;

  /* actions.c */
  struct {
    xine_stream_t           *stream;
    int                      start_pos;
    int                      start_time_in_secs;
    int                      update_mmk;
    int                      running;
  } play_data;
  filebrowser_t             *load_stream;
  filebrowser_t             *load_sub;
  int                        last_playback_speed;
  int                        last_step_time;

  struct {
    int                      (*get_names)(gGui_t *gui, const char **names, int max);
    xine_sarray_t           *avail;
    skins_locations_t       *default_skin;
    skins_locations_t       *current_skin;
    int                      num;
    int                      change_config_entry;
  } skins;

  struct {
    int                      fd;
    pthread_t                thread;
    FILE                    *fbk;
  } stdctl;

  /* NOTE:
   * 1. a failing xine_open () will wait up to 1 second for the stream event queues
   *    to process the error messages. Some old libxine may even wait infinitely.
   *    better defer event messages during such calls.
   * 2. drop exactly repeated messages.
   * 3. have at most 4 unanswered messages open.
   * 4. have at most 1 unanswered "more info" open. */
  struct {
    pthread_mutex_t          mutex;
    struct timeval           until;
    int                      level;
    uint8_t                  helipad[4];
    xitk_register_key_t      more_info_win;
    xitk_register_key_t      msg_windows[4];
    unsigned int             msg_index;
    int                      used;
    unsigned int             flags[4];
    unsigned int             last_flags;
    char                     msg[4][8192];
    char                     last_msg[8192];
  } no_messages;
};

extern gGui_t *gGui;

void set_window_type_start(gGui_t *gui, xitk_window_t *xwin);
#define set_window_type_start(gui, xwin) \
  xitk_window_set_wm_window_type (xwin, video_window_is_visible ((gui)->vwin) < 2 ? WINDOW_TYPE_NORMAL : WINDOW_TYPE_NONE)

void gui_load_window_pos (gGui_t *gui, const char *name, int *x, int *y);
void gui_save_window_pos (gGui_t *gui, const char *name, xitk_register_key_t key);

#ifdef _XITK_H_
typedef struct gui_new_window_s {
  gGui_t *gui;         /** << in */
  const char *title;   /** << in */
  const char *id;      /** << in: window pos register name */
  const char *skin;    /** << in: NULL if noskin only */
  const char *wfskin;  /** << in: window focus image (optional) */
  void (*adjust) (struct gui_new_window_s *nw); /** << in: func to call before opening window or NULL */
  xitk_rect_t wr;      /** << in: default pos/noskin size or 0, out: actual pos/size */
  xitk_window_t *xwin; /** << out */
  xitk_image_t *bg;    /** << out */
  xitk_widget_list_t *wl; /** << out */
} gui_new_window_t;
/** return -3 (skin error), -2 (window error), -1 (wrong args), 0 (noskin OK), 1 (skinned OK). */
int gui_window_new (gui_new_window_t *nw);
#endif

void gui_playlist_init (gGui_t *gui);
int gui_playlist_trylock (gGui_t *gui);
#define gui_playlist_trylock(_gui) pthread_mutex_trylock (&(_gui)->mmk_mutex)
void gui_playlist_lock (gGui_t *gui);
#define gui_playlist_lock(_gui) pthread_mutex_lock (&(_gui)->mmk_mutex)
void gui_playlist_unlock (gGui_t *gui);
#define gui_playlist_unlock(_gui) pthread_mutex_unlock (&(_gui)->mmk_mutex)
void gui_playlist_deinit (gGui_t *gui);
void gui_playlist_probe_start (gGui_t *gui);

#ifdef HAVE_XML_PARSER_REENTRANT
# define xml_parser_init_R(X,D,L,M) X = xml_parser_init_r ((D), (L), (M))
# define xml_parser_build_tree_R(X,T) xml_parser_build_tree_r ((X), (T))
# define xml_parser_finalize_R(X) xml_parser_finalize_r ((X))
#else
# define xml_parser_init_R(X,D,L,M) xml_parser_init ((D), (L), (M))
# define xml_parser_build_tree_R(X,T) xml_parser_build_tree ((T))
# define xml_parser_finalize_R(X) do {} while (0)
#endif

#endif

