/**********************************************************************
 * $count_primitives example -- PLI application using VPI routines
 *
 * C source to find and count all primitives in a design, beginning
 * with the top-level module and all levels of hierarchy below the top.
 *
 * Usage: $count_primitives;
 *
 * For the book, "The Verilog PLI Handbook" by Stuart Sutherland
 *  Book copyright 1999, Kluwer Academic Publishers, Norwell, MA, USA
 *   Contact: www.wkap.il
 *  Example copyright 1998, Sutherland HDL Inc, Portland, Oregon, USA
 *   Contact: www.sutherland.com or (503) 692-0898
 *********************************************************************/

#include <stdlib.h>    /* ANSI C standard library */
#include <stdio.h>     /* ANSI C standard input/output library */
#include "vpi_user.h"  /* IEEE 1364 PLI VPI routine library  */
#include "veriuser.h"  /* IEEE 1364 PLI TF routine library    
                          (using TF routines for simulation control) */

/* prototypes of the PLI application routines */
int PLIbook_CountPrims_compiletf(), PLIbook_CountPrims_calltf();
int PLIbook_find_child_mod(),  PLIbook_count_local_prims();

/**********************************************************************
 * VPI Registration Data
 *********************************************************************/
void PLIbook_CountPrims_register()
{
  s_vpi_systf_data tf_data;   /* allocate register data structure */
  tf_data.type      = vpiSysTask;
  tf_data.tfname    = "$count_primitives";
  tf_data.calltf    = PLIbook_CountPrims_calltf;
  tf_data.compiletf = PLIbook_CountPrims_compiletf;
  tf_data.sizetf    = NULL;

  vpi_register_systf(&tf_data);
}
/*********************************************************************/

/* include utility routines to work with tfargs */
#include "vpi_utilities.c"

/**********************************************************************
 * Compiletf application
 *********************************************************************/
int PLIbook_CountPrims_compiletf(char *user_data)
{
  vpiHandle systf_h, tfarg_itr, tfarg_h;

  systf_h = vpi_handle(vpiSysTfCall, NULL);
  if (systf_h == NULL) {
    vpi_printf("ERROR: $count_primitives could not obtain handle to systf call\n");
    tf_dofinish(); /* abort simulation */
    return(0);
  }
  tfarg_itr = vpi_iterate(vpiArgument, systf_h);
  if (tfarg_itr != NULL) {
    vpi_printf("WARNING: arguments to $count_primitives are ignored.\n");
    vpi_free_object(tfarg_itr); /* because not scanning until null */
  }
  return(0);
}

/**********************************************************************
 * calltf routine
 *********************************************************************/
int PLIbook_CountPrims_calltf(char *user_data)
{
  vpiHandle top_mod_itr, top_mod_h;
  int total_prims = 0;
  top_mod_itr = vpi_iterate(vpiModule, NULL); /*get top modules*/
  while (top_mod_h = vpi_scan(top_mod_itr)) {
    total_prims += PLIbook_find_child_mod(top_mod_h);
  }
  vpi_printf("Total number of primitives is %d\n\n", total_prims);
  return(0);
}

/**********************************************************************
 * Function to look for module instances in local scope.
 * THIS FUNCTION CALLS ITSELF RECURSIVELY.
 *********************************************************************/
int PLIbook_find_child_mod(vpiHandle this_mod_h)
{
  vpiHandle child_mod_itr, child_mod_h;
  int prims_in_child;
  
  prims_in_child = PLIbook_count_local_prims(this_mod_h);
  child_mod_itr = vpi_iterate(vpiModule, this_mod_h);
  if (child_mod_itr != NULL)
    while (child_mod_h = vpi_scan(child_mod_itr))
      prims_in_child += PLIbook_find_child_mod(child_mod_h);
  return(prims_in_child);
}

/**********************************************************************
 * Function to count primitives in local scope.
 *********************************************************************/
int PLIbook_count_local_prims(vpiHandle module_h) 
{
  vpiHandle prim_itr, prim_h;
  int prims_in_mod = 0;

  prim_itr = vpi_iterate(vpiPrimitive, module_h);
  if (prim_itr != NULL)
    while (prim_h = vpi_scan(prim_itr))
      prims_in_mod++;
  return (prims_in_mod);
}
/*********************************************************************/
