/*
 * SpanDSP - a series of DSP components for telephony
 *
 * t4_states.h - state tables for T.4 FAX image processing
 *
 * Written by Steve Underwood <steveu@coppice.org>
 *
 * Copyright (C) 2004 Steve Underwood
 *
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 2.1,
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: t4_states.h,v 1.7 2008/04/17 14:26:58 steveu Exp $
 */

/* Legitimate runs of zero bits which are the tail end of one code
   plus the start of the next code do not exceed 10 bits. */

/*
 * Note that these tables are ordered such that the
 * index into the table is known to be either the
 * run length, or (run length / 64) + a fixed offset.
 */
static const t4_run_table_entry_t t4_white_codes[] =
{
    { 8, 0x00AC,    0},         /* 0011 0101 */
    { 6, 0x0038,    1},         /* 0001 11 */
    { 4, 0x000E,    2},         /* 0111 */
    { 4, 0x0001,    3},         /* 1000 */
    { 4, 0x000D,    4},         /* 1011 */
    { 4, 0x0003,    5},         /* 1100 */
    { 4, 0x0007,    6},         /* 1110 */
    { 4, 0x000F,    7},         /* 1111 */
    { 5, 0x0019,    8},         /* 1001 1 */
    { 5, 0x0005,    9},         /* 1010 0 */
    { 5, 0x001C,   10},         /* 0011 1 */
    { 5, 0x0002,   11},         /* 0100 0 */
    { 6, 0x0004,   12},         /* 0010 00 */
    { 6, 0x0030,   13},         /* 0000 11 */
    { 6, 0x000B,   14},         /* 1101 00 */
    { 6, 0x002B,   15},         /* 1101 01 */
    { 6, 0x0015,   16},         /* 1010 10 */
    { 6, 0x0035,   17},         /* 1010 11 */
    { 7, 0x0072,   18},         /* 0100 111 */
    { 7, 0x0018,   19},         /* 0001 100 */
    { 7, 0x0008,   20},         /* 0001 000 */
    { 7, 0x0074,   21},         /* 0010 111 */
    { 7, 0x0060,   22},         /* 0000 011 */
    { 7, 0x0010,   23},         /* 0000 100 */
    { 7, 0x000A,   24},         /* 0101 000 */
    { 7, 0x006A,   25},         /* 0101 011 */
    { 7, 0x0064,   26},         /* 0010 011 */
    { 7, 0x0012,   27},         /* 0100 100 */
    { 7, 0x000C,   28},         /* 0011 000 */
    { 8, 0x0040,   29},         /* 0000 0010 */
    { 8, 0x00C0,   30},         /* 0000 0011 */
    { 8, 0x0058,   31},         /* 0001 1010 */
    { 8, 0x00D8,   32},         /* 0001 1011 */
    { 8, 0x0048,   33},         /* 0001 0010 */
    { 8, 0x00C8,   34},         /* 0001 0011 */
    { 8, 0x0028,   35},         /* 0001 0100 */
    { 8, 0x00A8,   36},         /* 0001 0101 */
    { 8, 0x0068,   37},         /* 0001 0110 */
    { 8, 0x00E8,   38},         /* 0001 0111 */
    { 8, 0x0014,   39},         /* 0010 1000 */
    { 8, 0x0094,   40},         /* 0010 1001 */
    { 8, 0x0054,   41},         /* 0010 1010 */
    { 8, 0x00D4,   42},         /* 0010 1011 */
    { 8, 0x0034,   43},         /* 0010 1100 */
    { 8, 0x00B4,   44},         /* 0010 1101 */
    { 8, 0x0020,   45},         /* 0000 0100 */
    { 8, 0x00A0,   46},         /* 0000 0101 */
    { 8, 0x0050,   47},         /* 0000 1010 */
    { 8, 0x00D0,   48},         /* 0000 1011 */
    { 8, 0x004A,   49},         /* 0101 0010 */
    { 8, 0x00CA,   50},         /* 0101 0011 */
    { 8, 0x002A,   51},         /* 0101 0100 */
    { 8, 0x00AA,   52},         /* 0101 0101 */
    { 8, 0x0024,   53},         /* 0010 0100 */
    { 8, 0x00A4,   54},         /* 0010 0101 */
    { 8, 0x001A,   55},         /* 0101 1000 */
    { 8, 0x009A,   56},         /* 0101 1001 */
    { 8, 0x005A,   57},         /* 0101 1010 */
    { 8, 0x00DA,   58},         /* 0101 1011 */
    { 8, 0x0052,   59},         /* 0100 1010 */
    { 8, 0x00D2,   60},         /* 0100 1011 */
    { 8, 0x004C,   61},         /* 0011 0010 */
    { 8, 0x00CC,   62},         /* 0011 0011 */
    { 8, 0x002C,   63},         /* 0011 0100 */
    { 5, 0x001B,   64},         /* 1101 1 */
    { 5, 0x0009,  128},         /* 1001 0 */
    { 6, 0x003A,  192},         /* 0101 11 */
    { 7, 0x0076,  256},         /* 0110 111 */
    { 8, 0x006C,  320},         /* 0011 0110 */
    { 8, 0x00EC,  384},         /* 0011 0111 */
    { 8, 0x0026,  448},         /* 0110 0100 */
    { 8, 0x00A6,  512},         /* 0110 0101 */
    { 8, 0x0016,  576},         /* 0110 1000 */
    { 8, 0x00E6,  640},         /* 0110 0111 */
    { 9, 0x0066,  704},         /* 0110 0110 0 */
    { 9, 0x0166,  768},         /* 0110 0110 1 */
    { 9, 0x0096,  832},         /* 0110 1001 0 */
    { 9, 0x0196,  896},         /* 0110 1001 1 */
    { 9, 0x0056,  960},         /* 0110 1010 0 */
    { 9, 0x0156, 1024},         /* 0110 1010 1 */
    { 9, 0x00D6, 1088},         /* 0110 1011 0 */
    { 9, 0x01D6, 1152},         /* 0110 1011 1 */
    { 9, 0x0036, 1216},         /* 0110 1100 0 */
    { 9, 0x0136, 1280},         /* 0110 1100 1 */
    { 9, 0x00B6, 1344},         /* 0110 1101 0 */
    { 9, 0x01B6, 1408},         /* 0110 1101 1 */
    { 9, 0x0032, 1472},         /* 0100 1100 0 */
    { 9, 0x0132, 1536},         /* 0100 1100 1 */
    { 9, 0x00B2, 1600},         /* 0100 1101 0 */
    { 6, 0x0006, 1664},         /* 0110 00 */
    { 9, 0x01B2, 1728},         /* 0100 1101 1 */
    {11, 0x0080, 1792},         /* 0000 0001 000 */
    {11, 0x0180, 1856},         /* 0000 0001 100 */
    {11, 0x0580, 1920},         /* 0000 0001 101 */
    {12, 0x0480, 1984},         /* 0000 0001 0010 */
    {12, 0x0C80, 2048},         /* 0000 0001 0011 */
    {12, 0x0280, 2112},         /* 0000 0001 0100 */
    {12, 0x0A80, 2176},         /* 0000 0001 0101 */
    {12, 0x0680, 2240},         /* 0000 0001 0110 */
    {12, 0x0E80, 2304},         /* 0000 0001 0111 */
    {12, 0x0380, 2368},         /* 0000 0001 1100 */
    {12, 0x0B80, 2432},         /* 0000 0001 1101 */
    {12, 0x0780, 2496},         /* 0000 0001 1110 */
    {12, 0x0F80, 2560},         /* 0000 0001 1111 */
};

static const t4_run_table_entry_t t4_black_codes[] =
{
    {10, 0x03B0,    0},         /* 0000 1101 11 */
    { 3, 0x0002,    1},         /* 010 */
    { 2, 0x0003,    2},         /* 11 */
    { 2, 0x0001,    3},         /* 10 */
    { 3, 0x0006,    4},         /* 011 */
    { 4, 0x000C,    5},         /* 0011 */
    { 4, 0x0004,    6},         /* 0010 */
    { 5, 0x0018,    7},         /* 0001 1 */
    { 6, 0x0028,    8},         /* 0001 01 */
    { 6, 0x0008,    9},         /* 0001 00 */
    { 7, 0x0010,   10},         /* 0000 100 */
    { 7, 0x0050,   11},         /* 0000 101 */
    { 7, 0x0070,   12},         /* 0000 111 */
    { 8, 0x0020,   13},         /* 0000 0100 */
    { 8, 0x00E0,   14},         /* 0000 0111 */
    { 9, 0x0030,   15},         /* 0000 1100 0 */
    {10, 0x03A0,   16},         /* 0000 0101 11 */
    {10, 0x0060,   17},         /* 0000 0110 00 */
    {10, 0x0040,   18},         /* 0000 0010 00 */
    {11, 0x0730,   19},         /* 0000 1100 111 */
    {11, 0x00B0,   20},         /* 0000 1101 000 */
    {11, 0x01B0,   21},         /* 0000 1101 100 */
    {11, 0x0760,   22},         /* 0000 0110 111 */
    {11, 0x00A0,   23},         /* 0000 0101 000 */
    {11, 0x0740,   24},         /* 0000 0010 111 */
    {11, 0x00C0,   25},         /* 0000 0011 000 */
    {12, 0x0530,   26},         /* 0000 1100 1010 */
    {12, 0x0D30,   27},         /* 0000 1100 1011 */
    {12, 0x0330,   28},         /* 0000 1100 1100 */
    {12, 0x0B30,   29},         /* 0000 1100 1101 */
    {12, 0x0160,   30},         /* 0000 0110 1000 */
    {12, 0x0960,   31},         /* 0000 0110 1001 */
    {12, 0x0560,   32},         /* 0000 0110 1010 */
    {12, 0x0D60,   33},         /* 0000 0110 1011 */
    {12, 0x04B0,   34},         /* 0000 1101 0010 */
    {12, 0x0CB0,   35},         /* 0000 1101 0011 */
    {12, 0x02B0,   36},         /* 0000 1101 0100 */
    {12, 0x0AB0,   37},         /* 0000 1101 0101 */
    {12, 0x06B0,   38},         /* 0000 1101 0110 */
    {12, 0x0EB0,   39},         /* 0000 1101 0111 */
    {12, 0x0360,   40},         /* 0000 0110 1100 */
    {12, 0x0B60,   41},         /* 0000 0110 1101 */
    {12, 0x05B0,   42},         /* 0000 1101 1010 */
    {12, 0x0DB0,   43},         /* 0000 1101 1011 */
    {12, 0x02A0,   44},         /* 0000 0101 0100 */
    {12, 0x0AA0,   45},         /* 0000 0101 0101 */
    {12, 0x06A0,   46},         /* 0000 0101 0110 */
    {12, 0x0EA0,   47},         /* 0000 0101 0111 */
    {12, 0x0260,   48},         /* 0000 0110 0100 */
    {12, 0x0A60,   49},         /* 0000 0110 0101 */
    {12, 0x04A0,   50},         /* 0000 0101 0010 */
    {12, 0x0CA0,   51},         /* 0000 0101 0011 */
    {12, 0x0240,   52},         /* 0000 0010 0100 */
    {12, 0x0EC0,   53},         /* 0000 0011 0111 */
    {12, 0x01C0,   54},         /* 0000 0011 1000 */
    {12, 0x0E40,   55},         /* 0000 0010 0111 */
    {12, 0x0140,   56},         /* 0000 0010 1000 */
    {12, 0x01A0,   57},         /* 0000 0101 1000 */
    {12, 0x09A0,   58},         /* 0000 0101 1001 */
    {12, 0x0D40,   59},         /* 0000 0010 1011 */
    {12, 0x0340,   60},         /* 0000 0010 1100 */
    {12, 0x05A0,   61},         /* 0000 0101 1010 */
    {12, 0x0660,   62},         /* 0000 0110 0110 */
    {12, 0x0E60,   63},         /* 0000 0110 0111 */
    {10, 0x03C0,   64},         /* 0000 0011 11 */
    {12, 0x0130,  128},         /* 0000 1100 1000 */
    {12, 0x0930,  192},         /* 0000 1100 1001 */
    {12, 0x0DA0,  256},         /* 0000 0101 1011 */
    {12, 0x0CC0,  320},         /* 0000 0011 0011 */
    {12, 0x02C0,  384},         /* 0000 0011 0100 */
    {12, 0x0AC0,  448},         /* 0000 0011 0101 */
    {13, 0x06C0,  512},         /* 0000 0011 0110 0 */
    {13, 0x16C0,  576},         /* 0000 0011 0110 1 */
    {13, 0x0A40,  640},         /* 0000 0010 0101 0 */
    {13, 0x1A40,  704},         /* 0000 0010 0101 1 */
    {13, 0x0640,  768},         /* 0000 0010 0110 0 */
    {13, 0x1640,  832},         /* 0000 0010 0110 1 */
    {13, 0x09C0,  896},         /* 0000 0011 1001 0 */
    {13, 0x19C0,  960},         /* 0000 0011 1001 1 */
    {13, 0x05C0, 1024},         /* 0000 0011 1010 0 */
    {13, 0x15C0, 1088},         /* 0000 0011 1010 1 */
    {13, 0x0DC0, 1152},         /* 0000 0011 1011 0 */
    {13, 0x1DC0, 1216},         /* 0000 0011 1011 1 */
    {13, 0x0940, 1280},         /* 0000 0010 1001 0 */
    {13, 0x1940, 1344},         /* 0000 0010 1001 1 */
    {13, 0x0540, 1408},         /* 0000 0010 1010 0 */
    {13, 0x1540, 1472},         /* 0000 0010 1010 1 */
    {13, 0x0B40, 1536},         /* 0000 0010 1101 0 */
    {13, 0x1B40, 1600},         /* 0000 0010 1101 1 */
    {13, 0x04C0, 1664},         /* 0000 0011 0010 0 */
    {13, 0x14C0, 1728},         /* 0000 0011 0010 1 */
    {11, 0x0080, 1792},         /* 0000 0001 000 */
    {11, 0x0180, 1856},         /* 0000 0001 100 */
    {11, 0x0580, 1920},         /* 0000 0001 101 */
    {12, 0x0480, 1984},         /* 0000 0001 0010 */
    {12, 0x0C80, 2048},         /* 0000 0001 0011 */
    {12, 0x0280, 2112},         /* 0000 0001 0100 */
    {12, 0x0A80, 2176},         /* 0000 0001 0101 */
    {12, 0x0680, 2240},         /* 0000 0001 0110 */
    {12, 0x0E80, 2304},         /* 0000 0001 0111 */
    {12, 0x0380, 2368},         /* 0000 0001 1100 */
    {12, 0x0B80, 2432},         /* 0000 0001 1101 */
    {12, 0x0780, 2496},         /* 0000 0001 1110 */
    {12, 0x0F80, 2560},         /* 0000 0001 1111 */
};

static const t4_table_entry_t t4_2d_table[128] =
{
    {S_Null,       1,    0},
    {S_Vert,       1,    0},
    {S_Vert,       3,   -1},
    {S_Vert,       1,    0},
    {S_Horiz,      3,    0},
    {S_Vert,       1,    0},
    {S_Vert,       3,    1},
    {S_Vert,       1,    0},
    {S_Pass,       4,    0},
    {S_Vert,       1,    0},
    {S_Vert,       3,   -1},
    {S_Vert,       1,    0},
    {S_Horiz,      3,    0},
    {S_Vert,       1,    0},
    {S_Vert,       3,    1},
    {S_Vert,       1,    0},
    {S_Vert,       6,   -2},
    {S_Vert,       1,    0},
    {S_Vert,       3,   -1},
    {S_Vert,       1,    0},
    {S_Horiz,      3,    0},
    {S_Vert,       1,    0},
    {S_Vert,       3,    1},
    {S_Vert,       1,    0},
    {S_Pass,       4,    0},
    {S_Vert,       1,    0},
    {S_Vert,       3,   -1},
    {S_Vert,       1,    0},
    {S_Horiz,      3,    0},
    {S_Vert,       1,    0},
    {S_Vert,       3,    1},
    {S_Vert,       1,    0},
    {S_Vert,       7,   -3},
    {S_Vert,       1,    0},
    {S_Vert,       3,   -1},
    {S_Vert,       1,    0},
    {S_Horiz,      3,    0},
    {S_Vert,       1,    0},
    {S_Vert,       3,    1},
    {S_Vert,       1,    0},
    {S_Pass,       4,    0},
    {S_Vert,       1,    0},
    {S_Vert,       3,   -1},
    {S_Vert,       1,    0},
    {S_Horiz,      3,    0},
    {S_Vert,       1,    0},
    {S_Vert,       3,    1},
    {S_Vert,       1,    0},
    {S_Vert,       6,    2},
    {S_Vert,       1,    0},
    {S_Vert,       3,   -1},
    {S_Vert,       1,    0},
    {S_Horiz,      3,    0},
    {S_Vert,       1,    0},
    {S_Vert,       3,    1},
    {S_Vert,       1,    0},
    {S_Pass,       4,    0},
    {S_Vert,       1,    0},
    {S_Vert,       3,   -1},
    {S_Vert,       1,    0},
    {S_Horiz,      3,    0},
    {S_Vert,       1,    0},
    {S_Vert,       3,    1},
    {S_Vert,       1,    0},
    {S_Ext,        7,    0},
    {S_Vert,       1,    0},
    {S_Vert,       3,   -1},
    {S_Vert,       1,    0},
    {S_Horiz,      3,    0},
    {S_Vert,       1,    0},
    {S_Vert,       3,    1},
    {S_Vert,       1,    0},
    {S_Pass,       4,    0},
    {S_Vert,       1,    0},
    {S_Vert,       3,   -1},
    {S_Vert,       1,    0},
    {S_Horiz,      3,    0},
    {S_Vert,       1,    0},
    {S_Vert,       3,    1},
    {S_Vert,       1,    0},
    {S_Vert,       6,   -2},
    {S_Vert,       1,    0},
    {S_Vert,       3,   -1},
    {S_Vert,       1,    0},
    {S_Horiz,      3,    0},
    {S_Vert,       1,    0},
    {S_Vert,       3,    1},
    {S_Vert,       1,    0},
    {S_Pass,       4,    0},
    {S_Vert,       1,    0},
    {S_Vert,       3,   -1},
    {S_Vert,       1,    0},
    {S_Horiz,      3,    0},
    {S_Vert,       1,    0},
    {S_Vert,       3,    1},
    {S_Vert,       1,    0},
    {S_Vert,       7,    3},
    {S_Vert,       1,    0},
    {S_Vert,       3,   -1},
    {S_Vert,       1,    0},
    {S_Horiz,      3,    0},
    {S_Vert,       1,    0},
    {S_Vert,       3,    1},
    {S_Vert,       1,    0},
    {S_Pass,       4,    0},
    {S_Vert,       1,    0},
    {S_Vert,       3,   -1},
    {S_Vert,       1,    0},
    {S_Horiz,      3,    0},
    {S_Vert,       1,    0},
    {S_Vert,       3,    1},
    {S_Vert,       1,    0},
    {S_Vert,       6,    2},
    {S_Vert,       1,    0},
    {S_Vert,       3,   -1},
    {S_Vert,       1,    0},
    {S_Horiz,      3,    0},
    {S_Vert,       1,    0},
    {S_Vert,       3,    1},
    {S_Vert,       1,    0},
    {S_Pass,       4,    0},
    {S_Vert,       1,    0},
    {S_Vert,       3,   -1},
    {S_Vert,       1,    0},
    {S_Horiz,      3,    0},
    {S_Vert,       1,    0},
    {S_Vert,       3,    1},
    {S_Vert,       1,    0}
};

static const t4_table_entry_t t4_1d_white_table[4096] =
{
    {S_Null,       1,    0},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      7,   20},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   24},
    {S_TermW,      6,   14},
    {S_TermW,      7,   28},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   23},
    {S_TermW,      4,    3},
    {S_TermW,      7,   27},
    {S_TermW,      4,    5},
    {S_TermW,      8,   39},
    {S_TermW,      6,   16},
    {S_MakeUpW,    8,  576},
    {S_TermW,      4,    6},
    {S_TermW,      7,   19},
    {S_TermW,      5,    8},
    {S_TermW,      8,   55},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   45},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      8,   53},
    {S_TermW,      5,    9},
    {S_MakeUpW,    8,  448},
    {S_TermW,      4,    6},
    {S_TermW,      8,   35},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   51},
    {S_TermW,      6,   15},
    {S_TermW,      8,   63},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_MakeUpW,    9, 1472},
    {S_TermW,      4,    5},
    {S_TermW,      8,   43},
    {S_TermW,      6,   17},
    {S_MakeUpW,    9, 1216},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   29},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      8,   33},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   49},
    {S_TermW,      6,   14},
    {S_TermW,      8,   61},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   47},
    {S_TermW,      4,    3},
    {S_TermW,      8,   59},
    {S_TermW,      4,    5},
    {S_TermW,      8,   41},
    {S_TermW,      6,   16},
    {S_MakeUpW,    9,  960},
    {S_TermW,      4,    6},
    {S_TermW,      8,   31},
    {S_TermW,      5,    8},
    {S_TermW,      8,   57},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   22},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      7,   26},
    {S_TermW,      5,    9},
    {S_MakeUpW,    9,  704},
    {S_TermW,      4,    6},
    {S_TermW,      8,   37},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   25},
    {S_TermW,      6,   15},
    {S_MakeUpW,    8,  320},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_TermW,      7,   18},
    {S_TermW,      4,    5},
    {S_TermW,      7,   21},
    {S_TermW,      6,   17},
    {S_MakeUpW,    7,  256},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_MakeUp,    11, 1792},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      7,   20},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   24},
    {S_TermW,      6,   14},
    {S_TermW,      7,   28},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   23},
    {S_TermW,      4,    3},
    {S_TermW,      7,   27},
    {S_TermW,      4,    5},
    {S_TermW,      8,   40},
    {S_TermW,      6,   16},
    {S_MakeUpW,    9,  832},
    {S_TermW,      4,    6},
    {S_TermW,      7,   19},
    {S_TermW,      5,    8},
    {S_TermW,      8,   56},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   46},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      8,   54},
    {S_TermW,      5,    9},
    {S_MakeUpW,    8,  512},
    {S_TermW,      4,    6},
    {S_TermW,      8,   36},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   52},
    {S_TermW,      6,   15},
    {S_TermW,      8,    0},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_MakeUpW,    9, 1600},
    {S_TermW,      4,    5},
    {S_TermW,      8,   44},
    {S_TermW,      6,   17},
    {S_MakeUpW,    9, 1344},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   30},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      8,   34},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   50},
    {S_TermW,      6,   14},
    {S_TermW,      8,   62},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   48},
    {S_TermW,      4,    3},
    {S_TermW,      8,   60},
    {S_TermW,      4,    5},
    {S_TermW,      8,   42},
    {S_TermW,      6,   16},
    {S_MakeUpW,    9, 1088},
    {S_TermW,      4,    6},
    {S_TermW,      8,   32},
    {S_TermW,      5,    8},
    {S_TermW,      8,   58},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   22},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      7,   26},
    {S_TermW,      5,    9},
    {S_MakeUpW,    8,  640},
    {S_TermW,      4,    6},
    {S_TermW,      8,   38},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   25},
    {S_TermW,      6,   15},
    {S_MakeUpW,    8,  384},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_TermW,      7,   18},
    {S_TermW,      4,    5},
    {S_TermW,      7,   21},
    {S_TermW,      6,   17},
    {S_MakeUpW,    7,  256},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_Null,       1,    0},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      7,   20},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   24},
    {S_TermW,      6,   14},
    {S_TermW,      7,   28},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   23},
    {S_TermW,      4,    3},
    {S_TermW,      7,   27},
    {S_TermW,      4,    5},
    {S_TermW,      8,   39},
    {S_TermW,      6,   16},
    {S_MakeUpW,    8,  576},
    {S_TermW,      4,    6},
    {S_TermW,      7,   19},
    {S_TermW,      5,    8},
    {S_TermW,      8,   55},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   45},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      8,   53},
    {S_TermW,      5,    9},
    {S_MakeUpW,    8,  448},
    {S_TermW,      4,    6},
    {S_TermW,      8,   35},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   51},
    {S_TermW,      6,   15},
    {S_TermW,      8,   63},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_MakeUpW,    9, 1536},
    {S_TermW,      4,    5},
    {S_TermW,      8,   43},
    {S_TermW,      6,   17},
    {S_MakeUpW,    9, 1280},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   29},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      8,   33},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   49},
    {S_TermW,      6,   14},
    {S_TermW,      8,   61},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   47},
    {S_TermW,      4,    3},
    {S_TermW,      8,   59},
    {S_TermW,      4,    5},
    {S_TermW,      8,   41},
    {S_TermW,      6,   16},
    {S_MakeUpW,    9, 1024},
    {S_TermW,      4,    6},
    {S_TermW,      8,   31},
    {S_TermW,      5,    8},
    {S_TermW,      8,   57},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   22},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      7,   26},
    {S_TermW,      5,    9},
    {S_MakeUpW,    9,  768},
    {S_TermW,      4,    6},
    {S_TermW,      8,   37},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   25},
    {S_TermW,      6,   15},
    {S_MakeUpW,    8,  320},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_TermW,      7,   18},
    {S_TermW,      4,    5},
    {S_TermW,      7,   21},
    {S_TermW,      6,   17},
    {S_MakeUpW,    7,  256},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_MakeUp,    11, 1856},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      7,   20},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   24},
    {S_TermW,      6,   14},
    {S_TermW,      7,   28},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   23},
    {S_TermW,      4,    3},
    {S_TermW,      7,   27},
    {S_TermW,      4,    5},
    {S_TermW,      8,   40},
    {S_TermW,      6,   16},
    {S_MakeUpW,    9,  896},
    {S_TermW,      4,    6},
    {S_TermW,      7,   19},
    {S_TermW,      5,    8},
    {S_TermW,      8,   56},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   46},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      8,   54},
    {S_TermW,      5,    9},
    {S_MakeUpW,    8,  512},
    {S_TermW,      4,    6},
    {S_TermW,      8,   36},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   52},
    {S_TermW,      6,   15},
    {S_TermW,      8,    0},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_MakeUpW,    9, 1728},
    {S_TermW,      4,    5},
    {S_TermW,      8,   44},
    {S_TermW,      6,   17},
    {S_MakeUpW,    9, 1408},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   30},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      8,   34},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   50},
    {S_TermW,      6,   14},
    {S_TermW,      8,   62},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   48},
    {S_TermW,      4,    3},
    {S_TermW,      8,   60},
    {S_TermW,      4,    5},
    {S_TermW,      8,   42},
    {S_TermW,      6,   16},
    {S_MakeUpW,    9, 1152},
    {S_TermW,      4,    6},
    {S_TermW,      8,   32},
    {S_TermW,      5,    8},
    {S_TermW,      8,   58},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   22},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      7,   26},
    {S_TermW,      5,    9},
    {S_MakeUpW,    8,  640},
    {S_TermW,      4,    6},
    {S_TermW,      8,   38},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   25},
    {S_TermW,      6,   15},
    {S_MakeUpW,    8,  384},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_TermW,      7,   18},
    {S_TermW,      4,    5},
    {S_TermW,      7,   21},
    {S_TermW,      6,   17},
    {S_MakeUpW,    7,  256},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_Null,       1,    0},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      7,   20},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   24},
    {S_TermW,      6,   14},
    {S_TermW,      7,   28},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   23},
    {S_TermW,      4,    3},
    {S_TermW,      7,   27},
    {S_TermW,      4,    5},
    {S_TermW,      8,   39},
    {S_TermW,      6,   16},
    {S_MakeUpW,    8,  576},
    {S_TermW,      4,    6},
    {S_TermW,      7,   19},
    {S_TermW,      5,    8},
    {S_TermW,      8,   55},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   45},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      8,   53},
    {S_TermW,      5,    9},
    {S_MakeUpW,    8,  448},
    {S_TermW,      4,    6},
    {S_TermW,      8,   35},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   51},
    {S_TermW,      6,   15},
    {S_TermW,      8,   63},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_MakeUpW,    9, 1472},
    {S_TermW,      4,    5},
    {S_TermW,      8,   43},
    {S_TermW,      6,   17},
    {S_MakeUpW,    9, 1216},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   29},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      8,   33},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   49},
    {S_TermW,      6,   14},
    {S_TermW,      8,   61},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   47},
    {S_TermW,      4,    3},
    {S_TermW,      8,   59},
    {S_TermW,      4,    5},
    {S_TermW,      8,   41},
    {S_TermW,      6,   16},
    {S_MakeUpW,    9,  960},
    {S_TermW,      4,    6},
    {S_TermW,      8,   31},
    {S_TermW,      5,    8},
    {S_TermW,      8,   57},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   22},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      7,   26},
    {S_TermW,      5,    9},
    {S_MakeUpW,    9,  704},
    {S_TermW,      4,    6},
    {S_TermW,      8,   37},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   25},
    {S_TermW,      6,   15},
    {S_MakeUpW,    8,  320},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_TermW,      7,   18},
    {S_TermW,      4,    5},
    {S_TermW,      7,   21},
    {S_TermW,      6,   17},
    {S_MakeUpW,    7,  256},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_MakeUp,    12, 2112},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      7,   20},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   24},
    {S_TermW,      6,   14},
    {S_TermW,      7,   28},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   23},
    {S_TermW,      4,    3},
    {S_TermW,      7,   27},
    {S_TermW,      4,    5},
    {S_TermW,      8,   40},
    {S_TermW,      6,   16},
    {S_MakeUpW,    9,  832},
    {S_TermW,      4,    6},
    {S_TermW,      7,   19},
    {S_TermW,      5,    8},
    {S_TermW,      8,   56},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   46},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      8,   54},
    {S_TermW,      5,    9},
    {S_MakeUpW,    8,  512},
    {S_TermW,      4,    6},
    {S_TermW,      8,   36},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   52},
    {S_TermW,      6,   15},
    {S_TermW,      8,    0},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_MakeUpW,    9, 1600},
    {S_TermW,      4,    5},
    {S_TermW,      8,   44},
    {S_TermW,      6,   17},
    {S_MakeUpW,    9, 1344},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   30},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      8,   34},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   50},
    {S_TermW,      6,   14},
    {S_TermW,      8,   62},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   48},
    {S_TermW,      4,    3},
    {S_TermW,      8,   60},
    {S_TermW,      4,    5},
    {S_TermW,      8,   42},
    {S_TermW,      6,   16},
    {S_MakeUpW,    9, 1088},
    {S_TermW,      4,    6},
    {S_TermW,      8,   32},
    {S_TermW,      5,    8},
    {S_TermW,      8,   58},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   22},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      7,   26},
    {S_TermW,      5,    9},
    {S_MakeUpW,    8,  640},
    {S_TermW,      4,    6},
    {S_TermW,      8,   38},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   25},
    {S_TermW,      6,   15},
    {S_MakeUpW,    8,  384},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_TermW,      7,   18},
    {S_TermW,      4,    5},
    {S_TermW,      7,   21},
    {S_TermW,      6,   17},
    {S_MakeUpW,    7,  256},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_Null,       1,    0},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      7,   20},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   24},
    {S_TermW,      6,   14},
    {S_TermW,      7,   28},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   23},
    {S_TermW,      4,    3},
    {S_TermW,      7,   27},
    {S_TermW,      4,    5},
    {S_TermW,      8,   39},
    {S_TermW,      6,   16},
    {S_MakeUpW,    8,  576},
    {S_TermW,      4,    6},
    {S_TermW,      7,   19},
    {S_TermW,      5,    8},
    {S_TermW,      8,   55},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   45},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      8,   53},
    {S_TermW,      5,    9},
    {S_MakeUpW,    8,  448},
    {S_TermW,      4,    6},
    {S_TermW,      8,   35},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   51},
    {S_TermW,      6,   15},
    {S_TermW,      8,   63},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_MakeUpW,    9, 1536},
    {S_TermW,      4,    5},
    {S_TermW,      8,   43},
    {S_TermW,      6,   17},
    {S_MakeUpW,    9, 1280},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   29},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      8,   33},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   49},
    {S_TermW,      6,   14},
    {S_TermW,      8,   61},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   47},
    {S_TermW,      4,    3},
    {S_TermW,      8,   59},
    {S_TermW,      4,    5},
    {S_TermW,      8,   41},
    {S_TermW,      6,   16},
    {S_MakeUpW,    9, 1024},
    {S_TermW,      4,    6},
    {S_TermW,      8,   31},
    {S_TermW,      5,    8},
    {S_TermW,      8,   57},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   22},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      7,   26},
    {S_TermW,      5,    9},
    {S_MakeUpW,    9,  768},
    {S_TermW,      4,    6},
    {S_TermW,      8,   37},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   25},
    {S_TermW,      6,   15},
    {S_MakeUpW,    8,  320},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_TermW,      7,   18},
    {S_TermW,      4,    5},
    {S_TermW,      7,   21},
    {S_TermW,      6,   17},
    {S_MakeUpW,    7,  256},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_MakeUp,    12, 2368},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      7,   20},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   24},
    {S_TermW,      6,   14},
    {S_TermW,      7,   28},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   23},
    {S_TermW,      4,    3},
    {S_TermW,      7,   27},
    {S_TermW,      4,    5},
    {S_TermW,      8,   40},
    {S_TermW,      6,   16},
    {S_MakeUpW,    9,  896},
    {S_TermW,      4,    6},
    {S_TermW,      7,   19},
    {S_TermW,      5,    8},
    {S_TermW,      8,   56},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   46},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      8,   54},
    {S_TermW,      5,    9},
    {S_MakeUpW,    8,  512},
    {S_TermW,      4,    6},
    {S_TermW,      8,   36},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   52},
    {S_TermW,      6,   15},
    {S_TermW,      8,    0},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_MakeUpW,    9, 1728},
    {S_TermW,      4,    5},
    {S_TermW,      8,   44},
    {S_TermW,      6,   17},
    {S_MakeUpW,    9, 1408},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   30},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      8,   34},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   50},
    {S_TermW,      6,   14},
    {S_TermW,      8,   62},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   48},
    {S_TermW,      4,    3},
    {S_TermW,      8,   60},
    {S_TermW,      4,    5},
    {S_TermW,      8,   42},
    {S_TermW,      6,   16},
    {S_MakeUpW,    9, 1152},
    {S_TermW,      4,    6},
    {S_TermW,      8,   32},
    {S_TermW,      5,    8},
    {S_TermW,      8,   58},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   22},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      7,   26},
    {S_TermW,      5,    9},
    {S_MakeUpW,    8,  640},
    {S_TermW,      4,    6},
    {S_TermW,      8,   38},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   25},
    {S_TermW,      6,   15},
    {S_MakeUpW,    8,  384},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_TermW,      7,   18},
    {S_TermW,      4,    5},
    {S_TermW,      7,   21},
    {S_TermW,      6,   17},
    {S_MakeUpW,    7,  256},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_Null,       1,    0},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      7,   20},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   24},
    {S_TermW,      6,   14},
    {S_TermW,      7,   28},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   23},
    {S_TermW,      4,    3},
    {S_TermW,      7,   27},
    {S_TermW,      4,    5},
    {S_TermW,      8,   39},
    {S_TermW,      6,   16},
    {S_MakeUpW,    8,  576},
    {S_TermW,      4,    6},
    {S_TermW,      7,   19},
    {S_TermW,      5,    8},
    {S_TermW,      8,   55},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   45},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      8,   53},
    {S_TermW,      5,    9},
    {S_MakeUpW,    8,  448},
    {S_TermW,      4,    6},
    {S_TermW,      8,   35},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   51},
    {S_TermW,      6,   15},
    {S_TermW,      8,   63},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_MakeUpW,    9, 1472},
    {S_TermW,      4,    5},
    {S_TermW,      8,   43},
    {S_TermW,      6,   17},
    {S_MakeUpW,    9, 1216},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   29},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      8,   33},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   49},
    {S_TermW,      6,   14},
    {S_TermW,      8,   61},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   47},
    {S_TermW,      4,    3},
    {S_TermW,      8,   59},
    {S_TermW,      4,    5},
    {S_TermW,      8,   41},
    {S_TermW,      6,   16},
    {S_MakeUpW,    9,  960},
    {S_TermW,      4,    6},
    {S_TermW,      8,   31},
    {S_TermW,      5,    8},
    {S_TermW,      8,   57},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   22},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      7,   26},
    {S_TermW,      5,    9},
    {S_MakeUpW,    9,  704},
    {S_TermW,      4,    6},
    {S_TermW,      8,   37},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   25},
    {S_TermW,      6,   15},
    {S_MakeUpW,    8,  320},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_TermW,      7,   18},
    {S_TermW,      4,    5},
    {S_TermW,      7,   21},
    {S_TermW,      6,   17},
    {S_MakeUpW,    7,  256},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_MakeUp,    12, 1984},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      7,   20},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   24},
    {S_TermW,      6,   14},
    {S_TermW,      7,   28},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   23},
    {S_TermW,      4,    3},
    {S_TermW,      7,   27},
    {S_TermW,      4,    5},
    {S_TermW,      8,   40},
    {S_TermW,      6,   16},
    {S_MakeUpW,    9,  832},
    {S_TermW,      4,    6},
    {S_TermW,      7,   19},
    {S_TermW,      5,    8},
    {S_TermW,      8,   56},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   46},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      8,   54},
    {S_TermW,      5,    9},
    {S_MakeUpW,    8,  512},
    {S_TermW,      4,    6},
    {S_TermW,      8,   36},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   52},
    {S_TermW,      6,   15},
    {S_TermW,      8,    0},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_MakeUpW,    9, 1600},
    {S_TermW,      4,    5},
    {S_TermW,      8,   44},
    {S_TermW,      6,   17},
    {S_MakeUpW,    9, 1344},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   30},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      8,   34},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   50},
    {S_TermW,      6,   14},
    {S_TermW,      8,   62},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   48},
    {S_TermW,      4,    3},
    {S_TermW,      8,   60},
    {S_TermW,      4,    5},
    {S_TermW,      8,   42},
    {S_TermW,      6,   16},
    {S_MakeUpW,    9, 1088},
    {S_TermW,      4,    6},
    {S_TermW,      8,   32},
    {S_TermW,      5,    8},
    {S_TermW,      8,   58},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   22},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      7,   26},
    {S_TermW,      5,    9},
    {S_MakeUpW,    8,  640},
    {S_TermW,      4,    6},
    {S_TermW,      8,   38},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   25},
    {S_TermW,      6,   15},
    {S_MakeUpW,    8,  384},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_TermW,      7,   18},
    {S_TermW,      4,    5},
    {S_TermW,      7,   21},
    {S_TermW,      6,   17},
    {S_MakeUpW,    7,  256},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_Null,       1,    0},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      7,   20},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   24},
    {S_TermW,      6,   14},
    {S_TermW,      7,   28},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   23},
    {S_TermW,      4,    3},
    {S_TermW,      7,   27},
    {S_TermW,      4,    5},
    {S_TermW,      8,   39},
    {S_TermW,      6,   16},
    {S_MakeUpW,    8,  576},
    {S_TermW,      4,    6},
    {S_TermW,      7,   19},
    {S_TermW,      5,    8},
    {S_TermW,      8,   55},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   45},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      8,   53},
    {S_TermW,      5,    9},
    {S_MakeUpW,    8,  448},
    {S_TermW,      4,    6},
    {S_TermW,      8,   35},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   51},
    {S_TermW,      6,   15},
    {S_TermW,      8,   63},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_MakeUpW,    9, 1536},
    {S_TermW,      4,    5},
    {S_TermW,      8,   43},
    {S_TermW,      6,   17},
    {S_MakeUpW,    9, 1280},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   29},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      8,   33},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   49},
    {S_TermW,      6,   14},
    {S_TermW,      8,   61},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   47},
    {S_TermW,      4,    3},
    {S_TermW,      8,   59},
    {S_TermW,      4,    5},
    {S_TermW,      8,   41},
    {S_TermW,      6,   16},
    {S_MakeUpW,    9, 1024},
    {S_TermW,      4,    6},
    {S_TermW,      8,   31},
    {S_TermW,      5,    8},
    {S_TermW,      8,   57},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   22},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      7,   26},
    {S_TermW,      5,    9},
    {S_MakeUpW,    9,  768},
    {S_TermW,      4,    6},
    {S_TermW,      8,   37},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   25},
    {S_TermW,      6,   15},
    {S_MakeUpW,    8,  320},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_TermW,      7,   18},
    {S_TermW,      4,    5},
    {S_TermW,      7,   21},
    {S_TermW,      6,   17},
    {S_MakeUpW,    7,  256},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_MakeUp,    11, 1920},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      7,   20},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   24},
    {S_TermW,      6,   14},
    {S_TermW,      7,   28},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   23},
    {S_TermW,      4,    3},
    {S_TermW,      7,   27},
    {S_TermW,      4,    5},
    {S_TermW,      8,   40},
    {S_TermW,      6,   16},
    {S_MakeUpW,    9,  896},
    {S_TermW,      4,    6},
    {S_TermW,      7,   19},
    {S_TermW,      5,    8},
    {S_TermW,      8,   56},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   46},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      8,   54},
    {S_TermW,      5,    9},
    {S_MakeUpW,    8,  512},
    {S_TermW,      4,    6},
    {S_TermW,      8,   36},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   52},
    {S_TermW,      6,   15},
    {S_TermW,      8,    0},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_MakeUpW,    9, 1728},
    {S_TermW,      4,    5},
    {S_TermW,      8,   44},
    {S_TermW,      6,   17},
    {S_MakeUpW,    9, 1408},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   30},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      8,   34},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   50},
    {S_TermW,      6,   14},
    {S_TermW,      8,   62},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   48},
    {S_TermW,      4,    3},
    {S_TermW,      8,   60},
    {S_TermW,      4,    5},
    {S_TermW,      8,   42},
    {S_TermW,      6,   16},
    {S_MakeUpW,    9, 1152},
    {S_TermW,      4,    6},
    {S_TermW,      8,   32},
    {S_TermW,      5,    8},
    {S_TermW,      8,   58},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   22},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      7,   26},
    {S_TermW,      5,    9},
    {S_MakeUpW,    8,  640},
    {S_TermW,      4,    6},
    {S_TermW,      8,   38},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   25},
    {S_TermW,      6,   15},
    {S_MakeUpW,    8,  384},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_TermW,      7,   18},
    {S_TermW,      4,    5},
    {S_TermW,      7,   21},
    {S_TermW,      6,   17},
    {S_MakeUpW,    7,  256},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_Null,       1,    0},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      7,   20},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   24},
    {S_TermW,      6,   14},
    {S_TermW,      7,   28},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   23},
    {S_TermW,      4,    3},
    {S_TermW,      7,   27},
    {S_TermW,      4,    5},
    {S_TermW,      8,   39},
    {S_TermW,      6,   16},
    {S_MakeUpW,    8,  576},
    {S_TermW,      4,    6},
    {S_TermW,      7,   19},
    {S_TermW,      5,    8},
    {S_TermW,      8,   55},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   45},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      8,   53},
    {S_TermW,      5,    9},
    {S_MakeUpW,    8,  448},
    {S_TermW,      4,    6},
    {S_TermW,      8,   35},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   51},
    {S_TermW,      6,   15},
    {S_TermW,      8,   63},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_MakeUpW,    9, 1472},
    {S_TermW,      4,    5},
    {S_TermW,      8,   43},
    {S_TermW,      6,   17},
    {S_MakeUpW,    9, 1216},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   29},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      8,   33},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   49},
    {S_TermW,      6,   14},
    {S_TermW,      8,   61},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   47},
    {S_TermW,      4,    3},
    {S_TermW,      8,   59},
    {S_TermW,      4,    5},
    {S_TermW,      8,   41},
    {S_TermW,      6,   16},
    {S_MakeUpW,    9,  960},
    {S_TermW,      4,    6},
    {S_TermW,      8,   31},
    {S_TermW,      5,    8},
    {S_TermW,      8,   57},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   22},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      7,   26},
    {S_TermW,      5,    9},
    {S_MakeUpW,    9,  704},
    {S_TermW,      4,    6},
    {S_TermW,      8,   37},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   25},
    {S_TermW,      6,   15},
    {S_MakeUpW,    8,  320},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_TermW,      7,   18},
    {S_TermW,      4,    5},
    {S_TermW,      7,   21},
    {S_TermW,      6,   17},
    {S_MakeUpW,    7,  256},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_MakeUp,    12, 2240},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      7,   20},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   24},
    {S_TermW,      6,   14},
    {S_TermW,      7,   28},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   23},
    {S_TermW,      4,    3},
    {S_TermW,      7,   27},
    {S_TermW,      4,    5},
    {S_TermW,      8,   40},
    {S_TermW,      6,   16},
    {S_MakeUpW,    9,  832},
    {S_TermW,      4,    6},
    {S_TermW,      7,   19},
    {S_TermW,      5,    8},
    {S_TermW,      8,   56},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   46},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      8,   54},
    {S_TermW,      5,    9},
    {S_MakeUpW,    8,  512},
    {S_TermW,      4,    6},
    {S_TermW,      8,   36},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   52},
    {S_TermW,      6,   15},
    {S_TermW,      8,    0},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_MakeUpW,    9, 1600},
    {S_TermW,      4,    5},
    {S_TermW,      8,   44},
    {S_TermW,      6,   17},
    {S_MakeUpW,    9, 1344},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   30},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      8,   34},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   50},
    {S_TermW,      6,   14},
    {S_TermW,      8,   62},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   48},
    {S_TermW,      4,    3},
    {S_TermW,      8,   60},
    {S_TermW,      4,    5},
    {S_TermW,      8,   42},
    {S_TermW,      6,   16},
    {S_MakeUpW,    9, 1088},
    {S_TermW,      4,    6},
    {S_TermW,      8,   32},
    {S_TermW,      5,    8},
    {S_TermW,      8,   58},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   22},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      7,   26},
    {S_TermW,      5,    9},
    {S_MakeUpW,    8,  640},
    {S_TermW,      4,    6},
    {S_TermW,      8,   38},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   25},
    {S_TermW,      6,   15},
    {S_MakeUpW,    8,  384},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_TermW,      7,   18},
    {S_TermW,      4,    5},
    {S_TermW,      7,   21},
    {S_TermW,      6,   17},
    {S_MakeUpW,    7,  256},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_Null,       1,    0},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      7,   20},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   24},
    {S_TermW,      6,   14},
    {S_TermW,      7,   28},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   23},
    {S_TermW,      4,    3},
    {S_TermW,      7,   27},
    {S_TermW,      4,    5},
    {S_TermW,      8,   39},
    {S_TermW,      6,   16},
    {S_MakeUpW,    8,  576},
    {S_TermW,      4,    6},
    {S_TermW,      7,   19},
    {S_TermW,      5,    8},
    {S_TermW,      8,   55},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   45},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      8,   53},
    {S_TermW,      5,    9},
    {S_MakeUpW,    8,  448},
    {S_TermW,      4,    6},
    {S_TermW,      8,   35},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   51},
    {S_TermW,      6,   15},
    {S_TermW,      8,   63},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_MakeUpW,    9, 1536},
    {S_TermW,      4,    5},
    {S_TermW,      8,   43},
    {S_TermW,      6,   17},
    {S_MakeUpW,    9, 1280},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   29},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      8,   33},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   49},
    {S_TermW,      6,   14},
    {S_TermW,      8,   61},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   47},
    {S_TermW,      4,    3},
    {S_TermW,      8,   59},
    {S_TermW,      4,    5},
    {S_TermW,      8,   41},
    {S_TermW,      6,   16},
    {S_MakeUpW,    9, 1024},
    {S_TermW,      4,    6},
    {S_TermW,      8,   31},
    {S_TermW,      5,    8},
    {S_TermW,      8,   57},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   22},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      7,   26},
    {S_TermW,      5,    9},
    {S_MakeUpW,    9,  768},
    {S_TermW,      4,    6},
    {S_TermW,      8,   37},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   25},
    {S_TermW,      6,   15},
    {S_MakeUpW,    8,  320},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_TermW,      7,   18},
    {S_TermW,      4,    5},
    {S_TermW,      7,   21},
    {S_TermW,      6,   17},
    {S_MakeUpW,    7,  256},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_MakeUp,    12, 2496},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      7,   20},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   24},
    {S_TermW,      6,   14},
    {S_TermW,      7,   28},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   23},
    {S_TermW,      4,    3},
    {S_TermW,      7,   27},
    {S_TermW,      4,    5},
    {S_TermW,      8,   40},
    {S_TermW,      6,   16},
    {S_MakeUpW,    9,  896},
    {S_TermW,      4,    6},
    {S_TermW,      7,   19},
    {S_TermW,      5,    8},
    {S_TermW,      8,   56},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   46},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      8,   54},
    {S_TermW,      5,    9},
    {S_MakeUpW,    8,  512},
    {S_TermW,      4,    6},
    {S_TermW,      8,   36},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   52},
    {S_TermW,      6,   15},
    {S_TermW,      8,    0},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_MakeUpW,    9, 1728},
    {S_TermW,      4,    5},
    {S_TermW,      8,   44},
    {S_TermW,      6,   17},
    {S_MakeUpW,    9, 1408},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   30},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      8,   34},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   50},
    {S_TermW,      6,   14},
    {S_TermW,      8,   62},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   48},
    {S_TermW,      4,    3},
    {S_TermW,      8,   60},
    {S_TermW,      4,    5},
    {S_TermW,      8,   42},
    {S_TermW,      6,   16},
    {S_MakeUpW,    9, 1152},
    {S_TermW,      4,    6},
    {S_TermW,      8,   32},
    {S_TermW,      5,    8},
    {S_TermW,      8,   58},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   22},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      7,   26},
    {S_TermW,      5,    9},
    {S_MakeUpW,    8,  640},
    {S_TermW,      4,    6},
    {S_TermW,      8,   38},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   25},
    {S_TermW,      6,   15},
    {S_MakeUpW,    8,  384},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_TermW,      7,   18},
    {S_TermW,      4,    5},
    {S_TermW,      7,   21},
    {S_TermW,      6,   17},
    {S_MakeUpW,    7,  256},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_EOL,       12,    0},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      7,   20},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   24},
    {S_TermW,      6,   14},
    {S_TermW,      7,   28},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   23},
    {S_TermW,      4,    3},
    {S_TermW,      7,   27},
    {S_TermW,      4,    5},
    {S_TermW,      8,   39},
    {S_TermW,      6,   16},
    {S_MakeUpW,    8,  576},
    {S_TermW,      4,    6},
    {S_TermW,      7,   19},
    {S_TermW,      5,    8},
    {S_TermW,      8,   55},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   45},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      8,   53},
    {S_TermW,      5,    9},
    {S_MakeUpW,    8,  448},
    {S_TermW,      4,    6},
    {S_TermW,      8,   35},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   51},
    {S_TermW,      6,   15},
    {S_TermW,      8,   63},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_MakeUpW,    9, 1472},
    {S_TermW,      4,    5},
    {S_TermW,      8,   43},
    {S_TermW,      6,   17},
    {S_MakeUpW,    9, 1216},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   29},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      8,   33},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   49},
    {S_TermW,      6,   14},
    {S_TermW,      8,   61},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   47},
    {S_TermW,      4,    3},
    {S_TermW,      8,   59},
    {S_TermW,      4,    5},
    {S_TermW,      8,   41},
    {S_TermW,      6,   16},
    {S_MakeUpW,    9,  960},
    {S_TermW,      4,    6},
    {S_TermW,      8,   31},
    {S_TermW,      5,    8},
    {S_TermW,      8,   57},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   22},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      7,   26},
    {S_TermW,      5,    9},
    {S_MakeUpW,    9,  704},
    {S_TermW,      4,    6},
    {S_TermW,      8,   37},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   25},
    {S_TermW,      6,   15},
    {S_MakeUpW,    8,  320},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_TermW,      7,   18},
    {S_TermW,      4,    5},
    {S_TermW,      7,   21},
    {S_TermW,      6,   17},
    {S_MakeUpW,    7,  256},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_MakeUp,    11, 1792},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      7,   20},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   24},
    {S_TermW,      6,   14},
    {S_TermW,      7,   28},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   23},
    {S_TermW,      4,    3},
    {S_TermW,      7,   27},
    {S_TermW,      4,    5},
    {S_TermW,      8,   40},
    {S_TermW,      6,   16},
    {S_MakeUpW,    9,  832},
    {S_TermW,      4,    6},
    {S_TermW,      7,   19},
    {S_TermW,      5,    8},
    {S_TermW,      8,   56},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   46},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      8,   54},
    {S_TermW,      5,    9},
    {S_MakeUpW,    8,  512},
    {S_TermW,      4,    6},
    {S_TermW,      8,   36},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   52},
    {S_TermW,      6,   15},
    {S_TermW,      8,    0},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_MakeUpW,    9, 1600},
    {S_TermW,      4,    5},
    {S_TermW,      8,   44},
    {S_TermW,      6,   17},
    {S_MakeUpW,    9, 1344},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   30},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      8,   34},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   50},
    {S_TermW,      6,   14},
    {S_TermW,      8,   62},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   48},
    {S_TermW,      4,    3},
    {S_TermW,      8,   60},
    {S_TermW,      4,    5},
    {S_TermW,      8,   42},
    {S_TermW,      6,   16},
    {S_MakeUpW,    9, 1088},
    {S_TermW,      4,    6},
    {S_TermW,      8,   32},
    {S_TermW,      5,    8},
    {S_TermW,      8,   58},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   22},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      7,   26},
    {S_TermW,      5,    9},
    {S_MakeUpW,    8,  640},
    {S_TermW,      4,    6},
    {S_TermW,      8,   38},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   25},
    {S_TermW,      6,   15},
    {S_MakeUpW,    8,  384},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_TermW,      7,   18},
    {S_TermW,      4,    5},
    {S_TermW,      7,   21},
    {S_TermW,      6,   17},
    {S_MakeUpW,    7,  256},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_Null,       1,    0},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      7,   20},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   24},
    {S_TermW,      6,   14},
    {S_TermW,      7,   28},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   23},
    {S_TermW,      4,    3},
    {S_TermW,      7,   27},
    {S_TermW,      4,    5},
    {S_TermW,      8,   39},
    {S_TermW,      6,   16},
    {S_MakeUpW,    8,  576},
    {S_TermW,      4,    6},
    {S_TermW,      7,   19},
    {S_TermW,      5,    8},
    {S_TermW,      8,   55},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   45},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      8,   53},
    {S_TermW,      5,    9},
    {S_MakeUpW,    8,  448},
    {S_TermW,      4,    6},
    {S_TermW,      8,   35},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   51},
    {S_TermW,      6,   15},
    {S_TermW,      8,   63},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_MakeUpW,    9, 1536},
    {S_TermW,      4,    5},
    {S_TermW,      8,   43},
    {S_TermW,      6,   17},
    {S_MakeUpW,    9, 1280},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   29},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      8,   33},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   49},
    {S_TermW,      6,   14},
    {S_TermW,      8,   61},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   47},
    {S_TermW,      4,    3},
    {S_TermW,      8,   59},
    {S_TermW,      4,    5},
    {S_TermW,      8,   41},
    {S_TermW,      6,   16},
    {S_MakeUpW,    9, 1024},
    {S_TermW,      4,    6},
    {S_TermW,      8,   31},
    {S_TermW,      5,    8},
    {S_TermW,      8,   57},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   22},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      7,   26},
    {S_TermW,      5,    9},
    {S_MakeUpW,    9,  768},
    {S_TermW,      4,    6},
    {S_TermW,      8,   37},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   25},
    {S_TermW,      6,   15},
    {S_MakeUpW,    8,  320},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_TermW,      7,   18},
    {S_TermW,      4,    5},
    {S_TermW,      7,   21},
    {S_TermW,      6,   17},
    {S_MakeUpW,    7,  256},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_MakeUp,    11, 1856},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      7,   20},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   24},
    {S_TermW,      6,   14},
    {S_TermW,      7,   28},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   23},
    {S_TermW,      4,    3},
    {S_TermW,      7,   27},
    {S_TermW,      4,    5},
    {S_TermW,      8,   40},
    {S_TermW,      6,   16},
    {S_MakeUpW,    9,  896},
    {S_TermW,      4,    6},
    {S_TermW,      7,   19},
    {S_TermW,      5,    8},
    {S_TermW,      8,   56},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   46},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      8,   54},
    {S_TermW,      5,    9},
    {S_MakeUpW,    8,  512},
    {S_TermW,      4,    6},
    {S_TermW,      8,   36},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   52},
    {S_TermW,      6,   15},
    {S_TermW,      8,    0},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_MakeUpW,    9, 1728},
    {S_TermW,      4,    5},
    {S_TermW,      8,   44},
    {S_TermW,      6,   17},
    {S_MakeUpW,    9, 1408},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   30},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      8,   34},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   50},
    {S_TermW,      6,   14},
    {S_TermW,      8,   62},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   48},
    {S_TermW,      4,    3},
    {S_TermW,      8,   60},
    {S_TermW,      4,    5},
    {S_TermW,      8,   42},
    {S_TermW,      6,   16},
    {S_MakeUpW,    9, 1152},
    {S_TermW,      4,    6},
    {S_TermW,      8,   32},
    {S_TermW,      5,    8},
    {S_TermW,      8,   58},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   22},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      7,   26},
    {S_TermW,      5,    9},
    {S_MakeUpW,    8,  640},
    {S_TermW,      4,    6},
    {S_TermW,      8,   38},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   25},
    {S_TermW,      6,   15},
    {S_MakeUpW,    8,  384},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_TermW,      7,   18},
    {S_TermW,      4,    5},
    {S_TermW,      7,   21},
    {S_TermW,      6,   17},
    {S_MakeUpW,    7,  256},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_Null,       1,    0},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      7,   20},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   24},
    {S_TermW,      6,   14},
    {S_TermW,      7,   28},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   23},
    {S_TermW,      4,    3},
    {S_TermW,      7,   27},
    {S_TermW,      4,    5},
    {S_TermW,      8,   39},
    {S_TermW,      6,   16},
    {S_MakeUpW,    8,  576},
    {S_TermW,      4,    6},
    {S_TermW,      7,   19},
    {S_TermW,      5,    8},
    {S_TermW,      8,   55},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   45},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      8,   53},
    {S_TermW,      5,    9},
    {S_MakeUpW,    8,  448},
    {S_TermW,      4,    6},
    {S_TermW,      8,   35},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   51},
    {S_TermW,      6,   15},
    {S_TermW,      8,   63},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_MakeUpW,    9, 1472},
    {S_TermW,      4,    5},
    {S_TermW,      8,   43},
    {S_TermW,      6,   17},
    {S_MakeUpW,    9, 1216},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   29},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      8,   33},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   49},
    {S_TermW,      6,   14},
    {S_TermW,      8,   61},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   47},
    {S_TermW,      4,    3},
    {S_TermW,      8,   59},
    {S_TermW,      4,    5},
    {S_TermW,      8,   41},
    {S_TermW,      6,   16},
    {S_MakeUpW,    9,  960},
    {S_TermW,      4,    6},
    {S_TermW,      8,   31},
    {S_TermW,      5,    8},
    {S_TermW,      8,   57},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   22},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      7,   26},
    {S_TermW,      5,    9},
    {S_MakeUpW,    9,  704},
    {S_TermW,      4,    6},
    {S_TermW,      8,   37},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   25},
    {S_TermW,      6,   15},
    {S_MakeUpW,    8,  320},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_TermW,      7,   18},
    {S_TermW,      4,    5},
    {S_TermW,      7,   21},
    {S_TermW,      6,   17},
    {S_MakeUpW,    7,  256},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_MakeUp,    12, 2176},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      7,   20},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   24},
    {S_TermW,      6,   14},
    {S_TermW,      7,   28},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   23},
    {S_TermW,      4,    3},
    {S_TermW,      7,   27},
    {S_TermW,      4,    5},
    {S_TermW,      8,   40},
    {S_TermW,      6,   16},
    {S_MakeUpW,    9,  832},
    {S_TermW,      4,    6},
    {S_TermW,      7,   19},
    {S_TermW,      5,    8},
    {S_TermW,      8,   56},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   46},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      8,   54},
    {S_TermW,      5,    9},
    {S_MakeUpW,    8,  512},
    {S_TermW,      4,    6},
    {S_TermW,      8,   36},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   52},
    {S_TermW,      6,   15},
    {S_TermW,      8,    0},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_MakeUpW,    9, 1600},
    {S_TermW,      4,    5},
    {S_TermW,      8,   44},
    {S_TermW,      6,   17},
    {S_MakeUpW,    9, 1344},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   30},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      8,   34},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   50},
    {S_TermW,      6,   14},
    {S_TermW,      8,   62},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   48},
    {S_TermW,      4,    3},
    {S_TermW,      8,   60},
    {S_TermW,      4,    5},
    {S_TermW,      8,   42},
    {S_TermW,      6,   16},
    {S_MakeUpW,    9, 1088},
    {S_TermW,      4,    6},
    {S_TermW,      8,   32},
    {S_TermW,      5,    8},
    {S_TermW,      8,   58},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   22},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      7,   26},
    {S_TermW,      5,    9},
    {S_MakeUpW,    8,  640},
    {S_TermW,      4,    6},
    {S_TermW,      8,   38},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   25},
    {S_TermW,      6,   15},
    {S_MakeUpW,    8,  384},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_TermW,      7,   18},
    {S_TermW,      4,    5},
    {S_TermW,      7,   21},
    {S_TermW,      6,   17},
    {S_MakeUpW,    7,  256},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_Null,       1,    0},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      7,   20},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   24},
    {S_TermW,      6,   14},
    {S_TermW,      7,   28},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   23},
    {S_TermW,      4,    3},
    {S_TermW,      7,   27},
    {S_TermW,      4,    5},
    {S_TermW,      8,   39},
    {S_TermW,      6,   16},
    {S_MakeUpW,    8,  576},
    {S_TermW,      4,    6},
    {S_TermW,      7,   19},
    {S_TermW,      5,    8},
    {S_TermW,      8,   55},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   45},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      8,   53},
    {S_TermW,      5,    9},
    {S_MakeUpW,    8,  448},
    {S_TermW,      4,    6},
    {S_TermW,      8,   35},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   51},
    {S_TermW,      6,   15},
    {S_TermW,      8,   63},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_MakeUpW,    9, 1536},
    {S_TermW,      4,    5},
    {S_TermW,      8,   43},
    {S_TermW,      6,   17},
    {S_MakeUpW,    9, 1280},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   29},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      8,   33},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   49},
    {S_TermW,      6,   14},
    {S_TermW,      8,   61},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   47},
    {S_TermW,      4,    3},
    {S_TermW,      8,   59},
    {S_TermW,      4,    5},
    {S_TermW,      8,   41},
    {S_TermW,      6,   16},
    {S_MakeUpW,    9, 1024},
    {S_TermW,      4,    6},
    {S_TermW,      8,   31},
    {S_TermW,      5,    8},
    {S_TermW,      8,   57},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   22},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      7,   26},
    {S_TermW,      5,    9},
    {S_MakeUpW,    9,  768},
    {S_TermW,      4,    6},
    {S_TermW,      8,   37},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   25},
    {S_TermW,      6,   15},
    {S_MakeUpW,    8,  320},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_TermW,      7,   18},
    {S_TermW,      4,    5},
    {S_TermW,      7,   21},
    {S_TermW,      6,   17},
    {S_MakeUpW,    7,  256},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_MakeUp,    12, 2432},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      7,   20},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   24},
    {S_TermW,      6,   14},
    {S_TermW,      7,   28},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   23},
    {S_TermW,      4,    3},
    {S_TermW,      7,   27},
    {S_TermW,      4,    5},
    {S_TermW,      8,   40},
    {S_TermW,      6,   16},
    {S_MakeUpW,    9,  896},
    {S_TermW,      4,    6},
    {S_TermW,      7,   19},
    {S_TermW,      5,    8},
    {S_TermW,      8,   56},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   46},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      8,   54},
    {S_TermW,      5,    9},
    {S_MakeUpW,    8,  512},
    {S_TermW,      4,    6},
    {S_TermW,      8,   36},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   52},
    {S_TermW,      6,   15},
    {S_TermW,      8,    0},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_MakeUpW,    9, 1728},
    {S_TermW,      4,    5},
    {S_TermW,      8,   44},
    {S_TermW,      6,   17},
    {S_MakeUpW,    9, 1408},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   30},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      8,   34},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   50},
    {S_TermW,      6,   14},
    {S_TermW,      8,   62},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   48},
    {S_TermW,      4,    3},
    {S_TermW,      8,   60},
    {S_TermW,      4,    5},
    {S_TermW,      8,   42},
    {S_TermW,      6,   16},
    {S_MakeUpW,    9, 1152},
    {S_TermW,      4,    6},
    {S_TermW,      8,   32},
    {S_TermW,      5,    8},
    {S_TermW,      8,   58},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   22},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      7,   26},
    {S_TermW,      5,    9},
    {S_MakeUpW,    8,  640},
    {S_TermW,      4,    6},
    {S_TermW,      8,   38},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   25},
    {S_TermW,      6,   15},
    {S_MakeUpW,    8,  384},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_TermW,      7,   18},
    {S_TermW,      4,    5},
    {S_TermW,      7,   21},
    {S_TermW,      6,   17},
    {S_MakeUpW,    7,  256},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_Null,       1,    0},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      7,   20},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   24},
    {S_TermW,      6,   14},
    {S_TermW,      7,   28},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   23},
    {S_TermW,      4,    3},
    {S_TermW,      7,   27},
    {S_TermW,      4,    5},
    {S_TermW,      8,   39},
    {S_TermW,      6,   16},
    {S_MakeUpW,    8,  576},
    {S_TermW,      4,    6},
    {S_TermW,      7,   19},
    {S_TermW,      5,    8},
    {S_TermW,      8,   55},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   45},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      8,   53},
    {S_TermW,      5,    9},
    {S_MakeUpW,    8,  448},
    {S_TermW,      4,    6},
    {S_TermW,      8,   35},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   51},
    {S_TermW,      6,   15},
    {S_TermW,      8,   63},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_MakeUpW,    9, 1472},
    {S_TermW,      4,    5},
    {S_TermW,      8,   43},
    {S_TermW,      6,   17},
    {S_MakeUpW,    9, 1216},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   29},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      8,   33},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   49},
    {S_TermW,      6,   14},
    {S_TermW,      8,   61},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   47},
    {S_TermW,      4,    3},
    {S_TermW,      8,   59},
    {S_TermW,      4,    5},
    {S_TermW,      8,   41},
    {S_TermW,      6,   16},
    {S_MakeUpW,    9,  960},
    {S_TermW,      4,    6},
    {S_TermW,      8,   31},
    {S_TermW,      5,    8},
    {S_TermW,      8,   57},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   22},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      7,   26},
    {S_TermW,      5,    9},
    {S_MakeUpW,    9,  704},
    {S_TermW,      4,    6},
    {S_TermW,      8,   37},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   25},
    {S_TermW,      6,   15},
    {S_MakeUpW,    8,  320},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_TermW,      7,   18},
    {S_TermW,      4,    5},
    {S_TermW,      7,   21},
    {S_TermW,      6,   17},
    {S_MakeUpW,    7,  256},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_MakeUp,    12, 2048},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      7,   20},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   24},
    {S_TermW,      6,   14},
    {S_TermW,      7,   28},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   23},
    {S_TermW,      4,    3},
    {S_TermW,      7,   27},
    {S_TermW,      4,    5},
    {S_TermW,      8,   40},
    {S_TermW,      6,   16},
    {S_MakeUpW,    9,  832},
    {S_TermW,      4,    6},
    {S_TermW,      7,   19},
    {S_TermW,      5,    8},
    {S_TermW,      8,   56},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   46},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      8,   54},
    {S_TermW,      5,    9},
    {S_MakeUpW,    8,  512},
    {S_TermW,      4,    6},
    {S_TermW,      8,   36},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   52},
    {S_TermW,      6,   15},
    {S_TermW,      8,    0},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_MakeUpW,    9, 1600},
    {S_TermW,      4,    5},
    {S_TermW,      8,   44},
    {S_TermW,      6,   17},
    {S_MakeUpW,    9, 1344},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   30},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      8,   34},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   50},
    {S_TermW,      6,   14},
    {S_TermW,      8,   62},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   48},
    {S_TermW,      4,    3},
    {S_TermW,      8,   60},
    {S_TermW,      4,    5},
    {S_TermW,      8,   42},
    {S_TermW,      6,   16},
    {S_MakeUpW,    9, 1088},
    {S_TermW,      4,    6},
    {S_TermW,      8,   32},
    {S_TermW,      5,    8},
    {S_TermW,      8,   58},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   22},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      7,   26},
    {S_TermW,      5,    9},
    {S_MakeUpW,    8,  640},
    {S_TermW,      4,    6},
    {S_TermW,      8,   38},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   25},
    {S_TermW,      6,   15},
    {S_MakeUpW,    8,  384},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_TermW,      7,   18},
    {S_TermW,      4,    5},
    {S_TermW,      7,   21},
    {S_TermW,      6,   17},
    {S_MakeUpW,    7,  256},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_Null,       1,    0},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      7,   20},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   24},
    {S_TermW,      6,   14},
    {S_TermW,      7,   28},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   23},
    {S_TermW,      4,    3},
    {S_TermW,      7,   27},
    {S_TermW,      4,    5},
    {S_TermW,      8,   39},
    {S_TermW,      6,   16},
    {S_MakeUpW,    8,  576},
    {S_TermW,      4,    6},
    {S_TermW,      7,   19},
    {S_TermW,      5,    8},
    {S_TermW,      8,   55},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   45},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      8,   53},
    {S_TermW,      5,    9},
    {S_MakeUpW,    8,  448},
    {S_TermW,      4,    6},
    {S_TermW,      8,   35},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   51},
    {S_TermW,      6,   15},
    {S_TermW,      8,   63},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_MakeUpW,    9, 1536},
    {S_TermW,      4,    5},
    {S_TermW,      8,   43},
    {S_TermW,      6,   17},
    {S_MakeUpW,    9, 1280},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   29},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      8,   33},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   49},
    {S_TermW,      6,   14},
    {S_TermW,      8,   61},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   47},
    {S_TermW,      4,    3},
    {S_TermW,      8,   59},
    {S_TermW,      4,    5},
    {S_TermW,      8,   41},
    {S_TermW,      6,   16},
    {S_MakeUpW,    9, 1024},
    {S_TermW,      4,    6},
    {S_TermW,      8,   31},
    {S_TermW,      5,    8},
    {S_TermW,      8,   57},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   22},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      7,   26},
    {S_TermW,      5,    9},
    {S_MakeUpW,    9,  768},
    {S_TermW,      4,    6},
    {S_TermW,      8,   37},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   25},
    {S_TermW,      6,   15},
    {S_MakeUpW,    8,  320},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_TermW,      7,   18},
    {S_TermW,      4,    5},
    {S_TermW,      7,   21},
    {S_TermW,      6,   17},
    {S_MakeUpW,    7,  256},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_MakeUp,    11, 1920},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      7,   20},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   24},
    {S_TermW,      6,   14},
    {S_TermW,      7,   28},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   23},
    {S_TermW,      4,    3},
    {S_TermW,      7,   27},
    {S_TermW,      4,    5},
    {S_TermW,      8,   40},
    {S_TermW,      6,   16},
    {S_MakeUpW,    9,  896},
    {S_TermW,      4,    6},
    {S_TermW,      7,   19},
    {S_TermW,      5,    8},
    {S_TermW,      8,   56},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   46},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      8,   54},
    {S_TermW,      5,    9},
    {S_MakeUpW,    8,  512},
    {S_TermW,      4,    6},
    {S_TermW,      8,   36},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   52},
    {S_TermW,      6,   15},
    {S_TermW,      8,    0},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_MakeUpW,    9, 1728},
    {S_TermW,      4,    5},
    {S_TermW,      8,   44},
    {S_TermW,      6,   17},
    {S_MakeUpW,    9, 1408},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   30},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      8,   34},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   50},
    {S_TermW,      6,   14},
    {S_TermW,      8,   62},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   48},
    {S_TermW,      4,    3},
    {S_TermW,      8,   60},
    {S_TermW,      4,    5},
    {S_TermW,      8,   42},
    {S_TermW,      6,   16},
    {S_MakeUpW,    9, 1152},
    {S_TermW,      4,    6},
    {S_TermW,      8,   32},
    {S_TermW,      5,    8},
    {S_TermW,      8,   58},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   22},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      7,   26},
    {S_TermW,      5,    9},
    {S_MakeUpW,    8,  640},
    {S_TermW,      4,    6},
    {S_TermW,      8,   38},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   25},
    {S_TermW,      6,   15},
    {S_MakeUpW,    8,  384},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_TermW,      7,   18},
    {S_TermW,      4,    5},
    {S_TermW,      7,   21},
    {S_TermW,      6,   17},
    {S_MakeUpW,    7,  256},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_Null,       1,    0},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      7,   20},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   24},
    {S_TermW,      6,   14},
    {S_TermW,      7,   28},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   23},
    {S_TermW,      4,    3},
    {S_TermW,      7,   27},
    {S_TermW,      4,    5},
    {S_TermW,      8,   39},
    {S_TermW,      6,   16},
    {S_MakeUpW,    8,  576},
    {S_TermW,      4,    6},
    {S_TermW,      7,   19},
    {S_TermW,      5,    8},
    {S_TermW,      8,   55},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   45},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      8,   53},
    {S_TermW,      5,    9},
    {S_MakeUpW,    8,  448},
    {S_TermW,      4,    6},
    {S_TermW,      8,   35},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   51},
    {S_TermW,      6,   15},
    {S_TermW,      8,   63},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_MakeUpW,    9, 1472},
    {S_TermW,      4,    5},
    {S_TermW,      8,   43},
    {S_TermW,      6,   17},
    {S_MakeUpW,    9, 1216},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   29},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      8,   33},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   49},
    {S_TermW,      6,   14},
    {S_TermW,      8,   61},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   47},
    {S_TermW,      4,    3},
    {S_TermW,      8,   59},
    {S_TermW,      4,    5},
    {S_TermW,      8,   41},
    {S_TermW,      6,   16},
    {S_MakeUpW,    9,  960},
    {S_TermW,      4,    6},
    {S_TermW,      8,   31},
    {S_TermW,      5,    8},
    {S_TermW,      8,   57},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   22},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      7,   26},
    {S_TermW,      5,    9},
    {S_MakeUpW,    9,  704},
    {S_TermW,      4,    6},
    {S_TermW,      8,   37},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   25},
    {S_TermW,      6,   15},
    {S_MakeUpW,    8,  320},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_TermW,      7,   18},
    {S_TermW,      4,    5},
    {S_TermW,      7,   21},
    {S_TermW,      6,   17},
    {S_MakeUpW,    7,  256},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_MakeUp,    12, 2304},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      7,   20},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   24},
    {S_TermW,      6,   14},
    {S_TermW,      7,   28},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   23},
    {S_TermW,      4,    3},
    {S_TermW,      7,   27},
    {S_TermW,      4,    5},
    {S_TermW,      8,   40},
    {S_TermW,      6,   16},
    {S_MakeUpW,    9,  832},
    {S_TermW,      4,    6},
    {S_TermW,      7,   19},
    {S_TermW,      5,    8},
    {S_TermW,      8,   56},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   46},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      8,   54},
    {S_TermW,      5,    9},
    {S_MakeUpW,    8,  512},
    {S_TermW,      4,    6},
    {S_TermW,      8,   36},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   52},
    {S_TermW,      6,   15},
    {S_TermW,      8,    0},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_MakeUpW,    9, 1600},
    {S_TermW,      4,    5},
    {S_TermW,      8,   44},
    {S_TermW,      6,   17},
    {S_MakeUpW,    9, 1344},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   30},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      8,   34},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   50},
    {S_TermW,      6,   14},
    {S_TermW,      8,   62},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   48},
    {S_TermW,      4,    3},
    {S_TermW,      8,   60},
    {S_TermW,      4,    5},
    {S_TermW,      8,   42},
    {S_TermW,      6,   16},
    {S_MakeUpW,    9, 1088},
    {S_TermW,      4,    6},
    {S_TermW,      8,   32},
    {S_TermW,      5,    8},
    {S_TermW,      8,   58},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   22},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      7,   26},
    {S_TermW,      5,    9},
    {S_MakeUpW,    8,  640},
    {S_TermW,      4,    6},
    {S_TermW,      8,   38},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   25},
    {S_TermW,      6,   15},
    {S_MakeUpW,    8,  384},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_TermW,      7,   18},
    {S_TermW,      4,    5},
    {S_TermW,      7,   21},
    {S_TermW,      6,   17},
    {S_MakeUpW,    7,  256},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_Null,       1,    0},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      7,   20},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   24},
    {S_TermW,      6,   14},
    {S_TermW,      7,   28},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   23},
    {S_TermW,      4,    3},
    {S_TermW,      7,   27},
    {S_TermW,      4,    5},
    {S_TermW,      8,   39},
    {S_TermW,      6,   16},
    {S_MakeUpW,    8,  576},
    {S_TermW,      4,    6},
    {S_TermW,      7,   19},
    {S_TermW,      5,    8},
    {S_TermW,      8,   55},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   45},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      8,   53},
    {S_TermW,      5,    9},
    {S_MakeUpW,    8,  448},
    {S_TermW,      4,    6},
    {S_TermW,      8,   35},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   51},
    {S_TermW,      6,   15},
    {S_TermW,      8,   63},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_MakeUpW,    9, 1536},
    {S_TermW,      4,    5},
    {S_TermW,      8,   43},
    {S_TermW,      6,   17},
    {S_MakeUpW,    9, 1280},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   29},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      8,   33},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   49},
    {S_TermW,      6,   14},
    {S_TermW,      8,   61},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   47},
    {S_TermW,      4,    3},
    {S_TermW,      8,   59},
    {S_TermW,      4,    5},
    {S_TermW,      8,   41},
    {S_TermW,      6,   16},
    {S_MakeUpW,    9, 1024},
    {S_TermW,      4,    6},
    {S_TermW,      8,   31},
    {S_TermW,      5,    8},
    {S_TermW,      8,   57},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   22},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      7,   26},
    {S_TermW,      5,    9},
    {S_MakeUpW,    9,  768},
    {S_TermW,      4,    6},
    {S_TermW,      8,   37},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   25},
    {S_TermW,      6,   15},
    {S_MakeUpW,    8,  320},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_TermW,      7,   18},
    {S_TermW,      4,    5},
    {S_TermW,      7,   21},
    {S_TermW,      6,   17},
    {S_MakeUpW,    7,  256},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_MakeUp,    12, 2560},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      7,   20},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   24},
    {S_TermW,      6,   14},
    {S_TermW,      7,   28},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   23},
    {S_TermW,      4,    3},
    {S_TermW,      7,   27},
    {S_TermW,      4,    5},
    {S_TermW,      8,   40},
    {S_TermW,      6,   16},
    {S_MakeUpW,    9,  896},
    {S_TermW,      4,    6},
    {S_TermW,      7,   19},
    {S_TermW,      5,    8},
    {S_TermW,      8,   56},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   46},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      8,   54},
    {S_TermW,      5,    9},
    {S_MakeUpW,    8,  512},
    {S_TermW,      4,    6},
    {S_TermW,      8,   36},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   52},
    {S_TermW,      6,   15},
    {S_TermW,      8,    0},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_MakeUpW,    9, 1728},
    {S_TermW,      4,    5},
    {S_TermW,      8,   44},
    {S_TermW,      6,   17},
    {S_MakeUpW,    9, 1408},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   30},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      8,   34},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   50},
    {S_TermW,      6,   14},
    {S_TermW,      8,   62},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   48},
    {S_TermW,      4,    3},
    {S_TermW,      8,   60},
    {S_TermW,      4,    5},
    {S_TermW,      8,   42},
    {S_TermW,      6,   16},
    {S_MakeUpW,    9, 1152},
    {S_TermW,      4,    6},
    {S_TermW,      8,   32},
    {S_TermW,      5,    8},
    {S_TermW,      8,   58},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   22},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      7,   26},
    {S_TermW,      5,    9},
    {S_MakeUpW,    8,  640},
    {S_TermW,      4,    6},
    {S_TermW,      8,   38},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   25},
    {S_TermW,      6,   15},
    {S_MakeUpW,    8,  384},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_TermW,      7,   18},
    {S_TermW,      4,    5},
    {S_TermW,      7,   21},
    {S_TermW,      6,   17},
    {S_MakeUpW,    7,  256},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7}
};

static const t4_table_entry_t t4_1d_black_table[8192] =
{
    {S_Null,       1,    0},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   13},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      9,   15},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     10,   18},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     10,   17},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUp,    11, 1792},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     11,   23},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     11,   20},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     11,   25},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   14},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_Null,       1,    0},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   13},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUpB,   12,  128},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   56},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   30},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUp,    11, 1856},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   57},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     11,   21},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   54},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   14},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_Null,       1,    0},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   13},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      9,   15},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   52},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   48},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUp,    12, 2112},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   44},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   36},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUpB,   12,  384},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   14},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_Null,       1,    0},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   13},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   28},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   60},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   40},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUp,    12, 2368},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     10,   16},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     10,    0},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUpB,   10,   64},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   14},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_Null,       1,    0},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   13},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      9,   15},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     10,   18},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     10,   17},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUp,    12, 1984},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   50},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   34},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUpB,   13, 1664},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   14},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_Null,       1,    0},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   13},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   26},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUpB,   13, 1408},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   32},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUp,    11, 1920},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   61},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   42},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUpB,   13, 1024},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   14},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_Null,       1,    0},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   13},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      9,   15},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUpB,   13,  768},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   62},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUp,    12, 2240},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   46},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   38},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUpB,   13,  512},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   14},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_Null,       1,    0},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   13},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     11,   19},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     11,   24},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     11,   22},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUp,    12, 2496},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     10,   16},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     10,    0},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUpB,   10,   64},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   14},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_EOL,       12,    0},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   13},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      9,   15},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     10,   18},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     10,   17},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUp,    11, 1792},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     11,   23},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     11,   20},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     11,   25},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   14},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_Null,       1,    0},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   13},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUpB,   12,  192},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUpB,   13, 1280},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   31},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUp,    11, 1856},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   58},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     11,   21},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUpB,   13,  896},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   14},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_Null,       1,    0},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   13},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      9,   15},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUpB,   13,  640},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   49},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUp,    12, 2176},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   45},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   37},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUpB,   12,  448},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   14},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_Null,       1,    0},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   13},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   29},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUpB,   13, 1536},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   41},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUp,    12, 2432},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     10,   16},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     10,    0},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUpB,   10,   64},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   14},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_Null,       1,    0},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   13},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      9,   15},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     10,   18},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     10,   17},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUp,    12, 2048},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   51},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   35},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUpB,   12,  320},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   14},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_Null,       1,    0},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   13},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   27},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   59},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   33},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUp,    11, 1920},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUpB,   12,  256},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   43},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUpB,   13, 1152},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   14},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_Null,       1,    0},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   13},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      9,   15},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   55},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   63},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUp,    12, 2304},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   47},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   39},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   53},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   14},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_Null,       1,    0},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   13},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     11,   19},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     11,   24},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     11,   22},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUp,    12, 2560},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     10,   16},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     10,    0},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUpB,   10,   64},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   14},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_Null,       1,    0},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   13},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      9,   15},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     10,   18},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     10,   17},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUp,    11, 1792},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     11,   23},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     11,   20},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     11,   25},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   14},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_Null,       1,    0},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   13},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUpB,   12,  128},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   56},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   30},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUp,    11, 1856},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   57},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     11,   21},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   54},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   14},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_Null,       1,    0},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   13},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      9,   15},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   52},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   48},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUp,    12, 2112},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   44},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   36},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUpB,   12,  384},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   14},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_Null,       1,    0},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   13},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   28},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   60},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   40},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUp,    12, 2368},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     10,   16},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     10,    0},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUpB,   10,   64},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   14},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_Null,       1,    0},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   13},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      9,   15},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     10,   18},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     10,   17},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUp,    12, 1984},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   50},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   34},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUpB,   13, 1728},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   14},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_Null,       1,    0},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   13},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   26},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUpB,   13, 1472},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   32},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUp,    11, 1920},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   61},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   42},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUpB,   13, 1088},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   14},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_Null,       1,    0},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   13},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      9,   15},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUpB,   13,  832},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   62},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUp,    12, 2240},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   46},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   38},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUpB,   13,  576},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   14},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_Null,       1,    0},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   13},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     11,   19},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     11,   24},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     11,   22},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUp,    12, 2496},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     10,   16},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     10,    0},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUpB,   10,   64},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   14},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_EOL,       12,    0},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   13},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      9,   15},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     10,   18},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     10,   17},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUp,    11, 1792},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     11,   23},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     11,   20},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     11,   25},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   14},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_Null,       1,    0},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   13},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUpB,   12,  192},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUpB,   13, 1344},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   31},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUp,    11, 1856},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   58},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     11,   21},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUpB,   13,  960},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   14},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_Null,       1,    0},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   13},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      9,   15},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUpB,   13,  704},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   49},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUp,    12, 2176},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   45},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   37},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUpB,   12,  448},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   14},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_Null,       1,    0},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   13},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   29},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUpB,   13, 1600},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   41},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUp,    12, 2432},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     10,   16},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     10,    0},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUpB,   10,   64},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   14},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_Null,       1,    0},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   13},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      9,   15},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     10,   18},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     10,   17},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUp,    12, 2048},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   51},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   35},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUpB,   12,  320},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   14},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_Null,       1,    0},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   13},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   27},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   59},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   33},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUp,    11, 1920},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUpB,   12,  256},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   43},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUpB,   13, 1216},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   14},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_Null,       1,    0},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   13},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      9,   15},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   55},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   63},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUp,    12, 2304},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   47},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   39},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   53},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   14},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_Null,       1,    0},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   13},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     11,   19},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     11,   24},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     11,   22},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUp,    12, 2560},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     10,   16},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     10,    0},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUpB,   10,   64},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   14},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2}
};

/*- End of file ------------------------------------------------------------*/
