/*****************************************************************************
 * $CAMITK_LICENCE_BEGIN$
 *
 * CamiTK - Computer Assisted Medical Intervention ToolKit
 * (c) 2001-2025 Univ. Grenoble Alpes, CNRS, Grenoble INP - UGA, TIMC, 38000 Grenoble, France
 *
 * Visit http://camitk.imag.fr for more information
 *
 * This file is part of CamiTK.
 *
 * CamiTK is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3
 * only, as published by the Free Software Foundation.
 *
 * CamiTK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * version 3 along with CamiTK.  If not, see <http://www.gnu.org/licenses/>.
 *
 * $CAMITK_LICENCE_END$
 ****************************************************************************/

#include "HotPlugActionExtension.h"
#include "CppHotPlugActionExtension.h"
#include "PythonHotPlugActionExtension.h"
#include "HotPlugAction.h"
#ifdef PYTHON_BINDING
#include "PythonManager.h"
#endif

#include <TransformEngine.h>
#include <CamiTKExtensionModel.h>
#include <Log.h>

//-- Qt Stuff
#include <QMessageBox>
#include <QApplication>

namespace camitk {

// -------------------- factory --------------------
HotPlugActionExtension* HotPlugActionExtension::newHotPlugActionExtension(const QString& camitkFilePath, bool forceVerifyOrRebuild) {
    CamiTKExtensionModel camitkExtensionModel;
    // check camitkExtensionModel
    if (camitkExtensionModel.load(camitkFilePath)) {
        const VariantDataModel& language = camitkExtensionModel.getModel()["language"];
        if (!language.isValid() || language.toString() == "C++") {

            return new CppHotPlugActionExtension(camitkFilePath, forceVerifyOrRebuild);
        }
        else {
#ifdef PYTHON_BINDING
            if (language.toString() == "Python" && !PythonManager::getPythonStatus().contains("Python disabled")) {
                return new PythonHotPlugActionExtension(camitkFilePath, forceVerifyOrRebuild);
            }
            else {
#endif
                CAMITK_WARNING_ALT(QString("CamiTK Extension file %1: language '%2' not supported or disabled.").arg(camitkFilePath).arg(language.toString()))
                return nullptr;
#ifdef PYTHON_BINDING
            }
#endif
        }

    }
    else {
        CAMITK_WARNING_ALT(QString("Creating extension: error in creating hot plug extension: cannot load file %1.").arg(camitkFilePath))
        return nullptr;
    }
}

// -------------------- constructor --------------------
HotPlugActionExtension::HotPlugActionExtension(const QString& camitkFilePath) : ActionExtension() {
    successfullyLoaded = false;
    CamiTKExtensionModel camitkExtensionModel(camitkFilePath);
    VariantDataModel& data = camitkExtensionModel.getModel();

    // start the engine
    TransformEngine transformEngine;
    name = transformEngine.transformToString("$title(name)$", QJsonObject::fromVariantMap(data.getValue().toMap()));
    description = data["description"];
    setLocation(camitkFilePath);
}

// -------------------- destructor --------------------
HotPlugActionExtension::~HotPlugActionExtension() {
}

// -------------------- getName --------------------
QString HotPlugActionExtension::getName() const {
    return name;
}

// -------------------- getDescription --------------------
QString HotPlugActionExtension::getDescription() const {
    return description;
}

// -------------------- isSuccessfullyLoaded --------------------
bool HotPlugActionExtension::isSuccessfullyLoaded() {
    return successfullyLoaded;
}

// -------------------- init --------------------
void HotPlugActionExtension::init() {
    initActions(); // as this ignore return value, initActions should be used instead
}

// -------------------- declaredActionCount --------------------
int HotPlugActionExtension::declaredActionCount() const {
    CamiTKExtensionModel camitkExtensionModel(getLocation());
    return camitkExtensionModel.getModel()["actions"].size();
}

} // namespace camitk